'use client';

import React, { useState } from 'react';
import { Form, Input, Button, Card, message, Space } from 'antd';
import { UserOutlined, LockOutlined, MailOutlined, UserAddOutlined } from '@ant-design/icons';
import { useRouter } from 'next/navigation';
import { fetcher } from '../../lib/fetcher';
import { RegisterRequest } from '../../types/auth';

export default function RegisterPage() {
  const [loading, setLoading] = useState(false);
  const [form] = Form.useForm();
  const [emailError, setEmailError] = useState<string>('');
  const [usernameError, setUsernameError] = useState<string>('');
  const router = useRouter();

  const handleRegister = async (values: RegisterRequest) => {
    // 清除之前的错误
    setEmailError('');
    setUsernameError('');
    
    setLoading(true);
    try {
      const response = await fetcher('/api/auth/register', {
        method: 'POST',
        body: JSON.stringify(values),
      });

      message.success('注册成功！');
      
      // 保存用户信息到localStorage
      localStorage.setItem('user', JSON.stringify({
        id: response.user.id,
        username: response.user.username,
        nickname: response.user.nickname,
        email: response.user.email,
        token: response.accessToken,
        refreshToken: response.refreshToken
      }));

      // 跳转到聊天页面
      router.push('/chat');
    } catch (error) {
      console.error('注册失败:', error);
      console.log('错误类型:', typeof error);
      console.log('错误消息:', error instanceof Error ? error.message : '非Error类型');
      if (error instanceof Error) {
        console.log('错误消息包含检查:');
        console.log('包含"用户名已存在":', error.message.includes('用户名已存在'));
        console.log('包含"邮箱已被注册":', error.message.includes('邮箱已被注册'));
        console.log('包含"admin用户名已被保留":', error.message.includes('admin用户名已被保留'));
        console.log('包含"admin昵称已被保留":', error.message.includes('admin昵称已被保留'));
        
        if (error.message.includes('用户名已存在')) {
          setUsernameError('用户名已存在，请选择其他用户名');
        } else if (error.message.includes('邮箱已被注册')) {
          setEmailError('邮箱已被注册，请使用其他邮箱');
        } else if (error.message.includes('admin用户名已被保留')) {
          setUsernameError('admin用户名已被保留，请选择其他用户名');
        } else if (error.message.includes('admin昵称已被保留')) {
          message.error('admin昵称已被保留，请选择其他昵称');
        } else {
          message.error(error.message || '注册失败，请重试');
        }
      } else {
        message.error('注册失败，请重试');
      }
    } finally {
      setLoading(false);
    }
  };

  const handleLogin = () => {
    router.push('/login');
  };

  return (
    <div className="font-sans flex items-center justify-center min-h-screen p-8 bg-gray-50">
      <main className="w-full max-w-md">
        <div className="text-center mb-8">
          <h1 className="text-3xl font-bold text-gray-800 mb-2">用户注册</h1>
          <p className="text-gray-600">创建您的聊天室账户</p>
        </div>

        <Card className="shadow-lg">
          <Form
            name="register"
            onFinish={handleRegister}
            autoComplete="off"
            layout="vertical"
            size="large"
          >
            <Form.Item
              name="username"
              label="用户名"
              rules={[
                { required: true, message: '请输入用户名' },
                { min: 3, max: 20, message: '用户名长度必须在3-20个字符之间' },
                { pattern: /^[a-zA-Z0-9_]+$/, message: '用户名只能包含字母、数字和下划线' },
                {
                  validator: async (_, value) => {
                    if (value && value.toLowerCase() === 'admin') {
                      throw new Error('admin用户名已被保留，请选择其他用户名');
                    }
                  }
                }
              ]}
            >
              <Input 
                prefix={<UserOutlined />} 
                placeholder="请输入用户名 (3-20字符，仅限字母、数字、下划线)"
                maxLength={20}
                showCount
                status={usernameError ? 'error' : undefined}
                onChange={() => setUsernameError('')}
              />
            </Form.Item>
            {usernameError && (
              <div className="text-red-500 text-xs mt-1 mb-4">
                {usernameError}
              </div>
            )}

            <Form.Item
              name="email"
              label="邮箱"
              rules={[
                { required: true, message: '请输入邮箱' },
                { type: 'email', message: '请输入有效的邮箱地址' }
              ]}
            >
              <Input 
                prefix={<MailOutlined />} 
                placeholder="请输入邮箱"
                type="email"
                status={emailError ? 'error' : undefined}
                onChange={() => setEmailError('')}
              />
            </Form.Item>
            {emailError && (
              <div className="text-red-500 text-xs mt-1 mb-4">
                {emailError}
              </div>
            )}

            <Form.Item
              name="nickname"
              label="昵称"
              rules={[
                { required: true, message: '请输入昵称' },
                { min: 2, max: 20, message: '昵称长度必须在2-20个字符之间' },
                {
                  validator: async (_, value) => {
                    if (value && value.toLowerCase() === 'admin') {
                      throw new Error('admin昵称已被保留，请选择其他昵称');
                    }
                  }
                }
              ]}
            >
              <Input 
                prefix={<UserOutlined />} 
                placeholder="请输入昵称 (2-20字符，支持中文、英文、数字)"
                maxLength={20}
                showCount
              />
            </Form.Item>

            <Form.Item
              name="password"
              label="密码"
              rules={[
                { required: true, message: '请输入密码' },
                { min: 6, message: '密码长度至少6个字符' },
                { pattern: /^(?=.*[a-zA-Z])(?=.*\d)/, message: '密码必须包含字母和数字' }
              ]}
            >
              <Input.Password 
                prefix={<LockOutlined />} 
                placeholder="请输入密码 (至少6位，必须包含字母和数字)"
                minLength={6}
              />
            </Form.Item>

            <Form.Item
              name="confirmPassword"
              label="确认密码"
              dependencies={['password']}
              rules={[
                { required: true, message: '请确认密码' },
                ({ getFieldValue }) => ({
                  validator(_, value) {
                    if (!value || getFieldValue('password') === value) {
                      return Promise.resolve();
                    }
                    return Promise.reject(new Error('两次输入的密码不一致'));
                  },
                }),
              ]}
            >
              <Input.Password 
                prefix={<LockOutlined />} 
                placeholder="请再次输入密码"
              />
            </Form.Item>

            <Form.Item>
              <Button 
                type="primary" 
                htmlType="submit" 
                loading={loading}
                icon={<UserAddOutlined />}
                className="w-full h-12"
              >
                注册
              </Button>
            </Form.Item>

            <div className="text-center">
              <Space>
                <span className="text-gray-600">已有账户？</span>
                <Button type="link" onClick={handleLogin} className="p-0">
                  立即登录
                </Button>
              </Space>
            </div>
          </Form>
        </Card>
      </main>
    </div>
  );
}
