'use client';

import React, { useState, useEffect } from 'react';
import { useRouter } from 'next/navigation';
import { Input, Button, Card, Typography, App } from 'antd';
import { UserOutlined, MessageOutlined } from '@ant-design/icons';
import styles from './page.module.css';

const { Title, Text } = Typography;

export default function NicknamePage() {
  const [nickname, setNickname] = useState('');
  const [error, setError] = useState('');
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [isClient, setIsClient] = useState(false);
  const router = useRouter();
  const { message } = App.useApp();

  // 客户端检测
  useEffect(() => {
    setIsClient(true);
  }, []);

  // 检查是否已经有昵称
  useEffect(() => {
    if (isClient) {
      const existingNickname = localStorage.getItem('nickname');
      if (existingNickname) {
        router.push('/chat');
      }
    }
  }, [isClient, router]);

  const handleEnterChat = async () => {
    if (!nickname.trim()) {
      setError('昵称不能为空');
      return;
    }

    if (nickname.trim().length < 2) {
      setError('昵称至少需要2个字符');
      return;
    }

    if (nickname.trim().length > 20) {
      setError('昵称不能超过20个字符');
      return;
    }

    setIsSubmitting(true);
    try {
      // 存储昵称到localStorage
      localStorage.setItem('nickname', nickname.trim());
      message.success('欢迎来到聊天室！');
      
      // 延迟跳转，让用户看到成功消息
      setTimeout(() => {
        router.push('/chat');
      }, 500);
    } catch (error) {
      console.error('设置昵称失败:', error);
      message.error('设置昵称失败，请重试');
    } finally {
      setIsSubmitting(false);
    }
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter') {
      handleEnterChat();
    }
  };

  // 在客户端渲染之前显示加载状态
  if (!isClient) {
    return (
      <div style={{ 
        display: 'flex', 
        justifyContent: 'center', 
        alignItems: 'center', 
        height: '100vh',
        fontSize: '16px',
        color: '#666'
      }}>
        加载中...
      </div>
    );
  }

  return (
    <main className={styles.main}>
      <div className={styles.container}>
        <div className={styles.logo}>
          <MessageOutlined className={styles.logoIcon} />
          <Title level={1} className={styles.title}>聊天室</Title>
        </div>
        
        <Card className={styles.card}>
          <div className={styles.cardHeader}>
            <UserOutlined className={styles.userIcon} />
            <Title level={3}>设置昵称</Title>
            <Text type="secondary">请输入您的昵称开始聊天</Text>
          </div>
          
          <Input
            size="large"
            placeholder="请输入您的昵称"
            value={nickname}
            onChange={(e) => {
              setNickname(e.target.value);
              if (error) setError('');
            }}
            onKeyPress={handleKeyPress}
            prefix={<UserOutlined />}
            className={styles.nicknameInput}
            maxLength={20}
            showCount
          />
          
          {error && (
            <div className={styles.errorMessage}>
              <Text type="danger">{error}</Text>
            </div>
          )}
          
          <Button
            type="primary"
            size="large"
            onClick={handleEnterChat}
            loading={isSubmitting}
            className={styles.enterButton}
            icon={<MessageOutlined />}
          >
            进入聊天室
          </Button>
        </Card>
        
        <div className={styles.footer}>
          <Text type="secondary">实时聊天，连接你我</Text>
        </div>
      </div>
    </main>
  );
}
