'use client';

import React, { useState, useEffect } from 'react';
import useSWR, { mutate } from 'swr';
import { List, Button, Spin, Alert, Modal, Input, App } from 'antd';
import { PlusOutlined, DeleteOutlined } from '@ant-design/icons';
import { fetcher } from '../lib/fetcher';
import { RoomPreviewInfo, RoomAddArgs, RoomDeleteArgs } from '../types/chat';
import styles from './RoomList.module.css';

interface RoomListProps {
  onSelectRoom: (roomId: number) => void;
  currentRoomId: number | null;
  nickname: string;
  user?: any;
  userRole?: 'guest' | 'user' | 'admin';
  onLogout?: () => void;
}

export default function RoomList({ onSelectRoom, currentRoomId, nickname, user, userRole = 'guest', onLogout }: RoomListProps) {
  const [isCreateModalVisible, setIsCreateModalVisible] = useState(false);
  const [newRoomName, setNewRoomName] = useState('');
  const [isCreating, setIsCreating] = useState(false);
  const [isClient, setIsClient] = useState(false);
  const [roomNameError, setRoomNameError] = useState<string>('');
  const { message, modal } = App.useApp();

  // 客户端检测
  useEffect(() => {
    setIsClient(true);
  }, []);

  const { data, error, isLoading } = useSWR<{ rooms: RoomPreviewInfo[] }>('/api/room/list', fetcher, {
    refreshInterval: 1000, // 每秒轮询一次
  });
  const rooms = data?.rooms || [];

  const handleCreateRoom = async () => {
    // 清除之前的错误
    setRoomNameError('');
    
    if (!newRoomName.trim()) {
      setRoomNameError('房间名称不能为空');
      return;
    }

    if (newRoomName.length < 2) {
      setRoomNameError('房间名称至少需要2个字符');
      return;
    }

    if (newRoomName.length > 50) {
      setRoomNameError('房间名称不能超过50个字符');
      return;
    }

    // 检查是否包含特殊字符
    if (!/^[\u4e00-\u9fa5a-zA-Z0-9_\s]+$/.test(newRoomName)) {
      setRoomNameError('房间名称只能包含中文、英文、数字、下划线和空格');
      return;
    }

    setIsCreating(true);
    try {
      const args: RoomAddArgs = {
        user: nickname,
        roomName: newRoomName.trim(),
      };

      const response = await fetcher('/api/room/add', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(args),
      });

      message.success('房间创建成功');
      setIsCreateModalVisible(false);
      setNewRoomName('');
      setRoomNameError('');
      
      // 刷新房间列表
      mutate('/api/room/list');
      
      // 自动进入新创建的房间
      if (response.roomId) {
        onSelectRoom(response.roomId);
      }
    } catch (error) {
      console.error('创建房间失败:', error);
      if (error instanceof Error) {
        if (error.message.includes('房间名称已存在')) {
          setRoomNameError('房间名称已存在，请选择其他名称');
        } else {
          setRoomNameError(error.message || '创建房间失败');
        }
      } else {
        setRoomNameError('创建房间失败');
      }
    } finally {
      setIsCreating(false);
    }
  };

  const handleDeleteRoom = async (roomId: number, event: React.MouseEvent) => {
    event.stopPropagation(); // 防止触发房间选择

    // 权限检查
    if (userRole === 'guest') {
      message.error('游客无法删除房间');
      return;
    }

    // 找到要删除的房间信息
    const roomToDelete = rooms.find(room => room.roomId === roomId);
    if (!roomToDelete) {
      message.error('房间不存在');
      return;
    }

    // 检查删除权限
    const isCreator = roomToDelete.creator === nickname;
    const isAdmin = userRole === 'admin';
    
    if (!isCreator && !isAdmin) {
      message.error('只有房间创建者或管理员可以删除房间');
      return;
    }

    modal.confirm({
      title: '确认删除',
      content: `确定要删除房间"${roomToDelete.roomName}"吗？删除后无法恢复。`,
      okText: '确认',
      cancelText: '取消',
      onOk: async () => {
        try {
          const args: RoomDeleteArgs = {
            user: nickname,
            roomId,
          };

          await fetcher('/api/room/delete', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(args),
          });

          message.success('房间删除成功');
          
          // 如果删除的是当前房间，清空选择
          if (currentRoomId === roomId) {
            onSelectRoom(0);
          }
          
          // 刷新房间列表
          mutate('/api/room/list');
        } catch (error) {
          console.error('删除房间失败:', error);
          message.error('删除房间失败');
        }
      },
    });
  };

  const handleRoomClick = (roomId: number) => {
    onSelectRoom(roomId);
  };

  if (!isClient) {
    return <div>加载中...</div>;
  }

  if (error) {
    return (
      <Alert
        message="加载失败"
        description="无法加载房间列表，请刷新页面重试"
        type="error"
        showIcon
      />
    );
  }

  return (
    <div className={styles.roomList}>
      <div className={styles.header}>
        <div className={styles.userInfo}>
          <h3>聊天房间</h3>
          {user && (
            <div className={styles.userDetails}>
              <span className={styles.username}>
                {user.username}
                {userRole === 'admin' && (
                  <span className={styles.adminBadge}>管理员</span>
                )}
              </span>
              <span className={styles.nickname}>({user.nickname})</span>
            </div>
          )}
          {!user && (
            <div className={styles.userDetails}>
              <span className={styles.guestMode}>游客模式</span>
              <span className={styles.nickname}>({nickname})</span>
            </div>
          )}
        </div>
        <div className={styles.headerActions}>
          {userRole !== 'guest' && (
            <Button
              type="primary"
              icon={<PlusOutlined />}
              onClick={() => setIsCreateModalVisible(true)}
              size="small"
            >
              创建房间
            </Button>
          )}
          {onLogout && (
            <Button
              type="text"
              size="small"
              onClick={onLogout}
              className={styles.logoutButton}
            >
              退出
            </Button>
          )}
        </div>
      </div>

      {isLoading ? (
        <div className={styles.loading}>
          <Spin size="large" />
        </div>
      ) : (
        <List
          className={styles.list}
          dataSource={rooms}
          renderItem={(room) => (
            <List.Item
              className={`${styles.roomItem} ${currentRoomId === room.roomId ? styles.active : ''}`}
              onClick={() => handleRoomClick(room.roomId)}
              actions={[
                // 只有非游客用户才显示删除按钮
                userRole !== 'guest' && (
                  <Button
                    key="delete"
                    type="text"
                    icon={<DeleteOutlined />}
                    size="small"
                    onClick={(e) => handleDeleteRoom(room.roomId, e)}
                    className={styles.deleteButton}
                  />
                ),
              ].filter(Boolean)}
            >
              <List.Item.Meta
                title={room.roomName}
                description={
                  room.lastMessage ? (
                    <div className={styles.lastMessage}>
                      <span className={styles.sender}>{room.lastMessage.sender}:</span>
                      <span className={styles.content}>{room.lastMessage.content}</span>
                    </div>
                  ) : (
                    '暂无消息'
                  )
                }
              />
            </List.Item>
          )}
        />
      )}

      <Modal
        title="创建新房间"
        open={isCreateModalVisible}
        onOk={handleCreateRoom}
        onCancel={() => {
          setIsCreateModalVisible(false);
          setNewRoomName('');
          setRoomNameError('');
        }}
        confirmLoading={isCreating}
        okText="创建"
        cancelText="取消"
      >
        <div className="space-y-4">
          <div>
            <Input
              placeholder="请输入房间名称 (2-50字符，支持中文、英文、数字、下划线、空格)"
              value={newRoomName}
              onChange={(e) => {
                setNewRoomName(e.target.value);
                setRoomNameError(''); // 清除错误提示
              }}
              onPressEnter={handleCreateRoom}
              autoFocus
              maxLength={50}
              showCount
              status={roomNameError ? 'error' : undefined}
            />
            {roomNameError && (
              <div className="text-red-500 text-xs mt-1">
                {roomNameError}
              </div>
            )}
          </div>
        </div>
      </Modal>
    </div>
  );
}
