'use client';

import React, { useEffect, useRef, useState } from 'react';
import useSWR, { mutate } from 'swr';
import { Spin, Alert, Typography, Empty } from 'antd';
import { fetcher } from '../lib/fetcher';
import { Message, RoomPreviewInfo } from '../types/chat';
import MessageItem from './MessageItem';
import MessageInput from './MessageInput';
import styles from './ChatWindow.module.css';

const { Title } = Typography;

interface ChatWindowProps {
  roomId: number | null;
  user?: any;
  userRole?: 'guest' | 'user' | 'admin';
}

export default function ChatWindow({ roomId, user, userRole = 'guest' }: ChatWindowProps) {
  const messageListRef = useRef<HTMLDivElement>(null);
  const [nickname, setNickname] = useState<string | null>(null);
  const [isClient, setIsClient] = useState(false);

  // 客户端检测
  useEffect(() => {
    setIsClient(true);
  }, []);

  // 获取昵称
  useEffect(() => {
    if (isClient) {
      if (user) {
        // 使用登录用户的昵称
        setNickname(user.nickname);
      } else {
        // 使用游客昵称
        const storedNickname = localStorage.getItem('nickname');
        setNickname(storedNickname);
      }
    }
  }, [isClient, user]);

  // 获取房间信息
  const { data: roomData } = useSWR<{ rooms: RoomPreviewInfo[] }>('/api/room/list', fetcher);
  const currentRoom = roomData?.rooms?.find(room => room.roomId === roomId);

  // 获取消息列表
  const {
    data: messageData,
    error,
    isLoading,
  } = useSWR<{ messages: Message[] }>(
    roomId ? `/api/room/message/list?roomId=${roomId}` : null,
    fetcher,
    {
      refreshInterval: 1000, // 每秒轮询一次
    }
  );
  const messages = messageData?.messages || [];

  // SSE 订阅：根据当前房间建立事件流
  useEffect(() => {
    if (!isClient || !roomId) return;

    const es = new EventSource(`/api/room/message/stream?roomId=${roomId}`);

    es.onmessage = (event) => {
      try {
        const payload = JSON.parse(event.data);
        if (payload?.type === 'message' && payload?.payload) {
          const incoming = payload.payload as Message;
          // 增量更新：若已存在相同 messageId 则忽略
          mutate(
            `/api/room/message/list?roomId=${roomId}`,
            (current: { messages?: Message[] } | undefined) => {
              const currentList = current?.messages || [];
              if (currentList.some((m) => m.messageId === incoming.messageId)) {
                return current;
              }
              return { messages: [...currentList, incoming] };
            },
            false
          );
        }
      } catch (e) {
        console.error('解析SSE消息失败:', e);
      }
    };

    es.onerror = (err) => {
      console.error('SSE连接错误:', err);
      // 出错时关闭，由浏览器/页面切换策略决定是否重连
      es.close();
    };

    return () => {
      es.close();
    };
  }, [isClient, roomId]);

  // 自动滚动到底部
  useEffect(() => {
    if (messageListRef.current) {
      messageListRef.current.scrollTop = messageListRef.current.scrollHeight;
    }
  }, [messages]);

  const handleSendMessage = async (content: string) => {
    if (!roomId || !nickname) return;

    const optimisticNewMessage: Message = {
      messageId: Date.now(), // 临时ID
      roomId,
      sender: nickname,
      content,
      time: Date.now(),
    };

    // 乐观更新UI
    mutate(
      (currentData: { messages?: Message[] } | undefined) => ({
        ...currentData,
        messages: [...(currentData?.messages || []), optimisticNewMessage]
      }),
      false
    );

    try {
      await fetcher('/api/message/add', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          roomId,
          sender: nickname,
          content,
        }),
      });
      
      // 触发重新获取以获取服务器端的实际消息
      mutate(`/api/room/message/list?roomId=${roomId}`);
    } catch (e) {
      console.error('发送消息失败:', e);
      // 发送失败时回滚乐观更新
      mutate(
        (currentData: { messages?: Message[] } | undefined) => ({
          ...currentData,
          messages: (currentData?.messages || []).filter((msg: Message) => msg.messageId !== optimisticNewMessage.messageId)
        }),
        false
      );
    }
  };

  if (!isClient) {
    return <div>加载中...</div>;
  }

  if (!roomId) {
    return (
      <div className={styles.emptyState}>
        <Empty
          description="请选择一个聊天房间"
          image={Empty.PRESENTED_IMAGE_SIMPLE}
        />
      </div>
    );
  }

  if (error) {
    return (
      <Alert
        message="加载失败"
        description="无法加载消息，请刷新页面重试"
        type="error"
        showIcon
        className={styles.errorAlert}
      />
    );
  }

  return (
    <div className={styles.chatWindow}>
      {/* 房间标题 */}
      <div className={styles.roomHeader}>
        <Title level={4} className={styles.roomTitle}>
          {currentRoom?.roomName || `房间 ${roomId}`}
        </Title>
        <div className={styles.roomInfo}>
          <span className={styles.messageCount}>
            {messages.length} 条消息
          </span>
        </div>
      </div>

      {/* 消息列表 */}
      <div className={styles.messageList} ref={messageListRef}>
        {isLoading ? (
          <div className={styles.loading}>
            <Spin size="large" />
          </div>
        ) : messages.length === 0 ? (
          <div className={styles.emptyMessages}>
            <Empty
              description="暂无消息"
              image={Empty.PRESENTED_IMAGE_SIMPLE}
            />
          </div>
        ) : (
          messages.map((message) => (
            <MessageItem
              key={message.messageId}
              message={message}
              isOwn={message.sender === nickname}
            />
          ))
        )}
      </div>

      {/* 消息输入框 */}
      <div className={styles.messageInput}>
        <MessageInput
          onSendMessage={handleSendMessage}
          disabled={!nickname}
          placeholder={nickname ? "输入消息..." : "请先设置昵称"}
        />
      </div>
    </div>
  );
}
