'use client';

import React, { useState, useEffect } from 'react';
import { Input, Button, Card, message, Space } from 'antd';
import { UserOutlined, MessageOutlined, LoginOutlined, UserAddOutlined } from '@ant-design/icons';
import { useRouter } from 'next/navigation';

export default function Home() {
  const [nickname, setNickname] = useState<string>('');
  const [isClient, setIsClient] = useState(false);
  const router = useRouter();

  // 客户端检测
  useEffect(() => {
    setIsClient(true);
  }, []);

  // 获取已保存的昵称
  useEffect(() => {
    if (isClient) {
      const savedNickname = localStorage.getItem('nickname');
      if (savedNickname) {
        setNickname(savedNickname);
      }
    }
  }, [isClient]);

  const handleSetNickname = () => {
    if (!nickname.trim()) {
      message.error('请输入昵称');
      return;
    }
    
    if (nickname.length < 2 || nickname.length > 20) {
      message.error('昵称长度必须在2-20个字符之间');
      return;
    }

    // 检查是否为admin昵称
    if (nickname.toLowerCase() === 'admin') {
      message.error('admin昵称已被保留，请选择其他昵称');
      return;
    }

    localStorage.setItem('nickname', nickname.trim());
    message.success('昵称设置成功！');
  };

  const handleEnterChat = () => {
    if (!nickname.trim()) {
      message.error('请先设置昵称');
      return;
    }

    // 检查是否为admin昵称
    if (nickname.toLowerCase() === 'admin') {
      message.error('admin昵称已被保留，请选择其他昵称');
      return;
    }

    router.push('/chat');
  };

  const handleLogin = () => {
    router.push('/login');
  };

  const handleRegister = () => {
    router.push('/register');
  };

  if (!isClient) {
    return <div>加载中...</div>;
  }

  return (
    <div className="font-sans flex items-center justify-center min-h-screen p-8 bg-gray-50">
      <main className="w-full max-w-md">
        <div className="text-center mb-8">
          <h1 className="text-3xl font-bold text-gray-800 mb-2">聊天室</h1>
          <p className="text-gray-600">选择您的登录方式</p>
        </div>

        <Card className="shadow-lg mb-4">
          <div className="space-y-4">
            <Button 
              type="primary" 
              size="large" 
              icon={<LoginOutlined />} 
              onClick={handleLogin}
              className="w-full h-12"
            >
              用户登录
            </Button>
            <Button 
              size="large" 
              icon={<UserAddOutlined />} 
              onClick={handleRegister}
              className="w-full h-12"
            >
              用户注册
            </Button>
          </div>
        </Card>

        <Card className="shadow-lg">
          <div className="text-center mb-4">
            <h3 className="text-lg font-medium text-gray-700">或使用昵称快速进入</h3>
          </div>
          <div className="space-y-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">昵称</label>
              <Input
                prefix={<UserOutlined />}
                placeholder="请输入昵称 (2-20字符)"
                value={nickname}
                onChange={(e) => setNickname(e.target.value)}
                onPressEnter={handleSetNickname}
                maxLength={20}
                showCount
                size="large"
              />
            </div>
            
            <Button
              type="default"
              size="large"
              icon={<MessageOutlined />}
              onClick={handleEnterChat}
              disabled={!nickname.trim()}
              className="w-full h-12"
            >
              快速进入聊天室
            </Button>
            
            {!nickname.trim() && (
              <p className="text-sm text-gray-500 text-center">
                请先设置昵称
              </p>
            )}
          </div>
        </Card>
      </main>
    </div>
  );
}
