'use client';

import React, { useState, useEffect } from 'react';
import { Layout, message } from 'antd';
import { useRouter } from 'next/navigation';
import RoomList from '../../components/RoomList';
import ChatWindow from '../../components/ChatWindow';
import styles from './page.module.css';

const { Sider, Content } = Layout;

export default function ChatPage() {
  const [currentRoomId, setCurrentRoomId] = useState<number | null>(null);
  const [nickname, setNickname] = useState<string | null>(null);
  const [user, setUser] = useState<any>(null);
  const [userRole, setUserRole] = useState<'guest' | 'user' | 'admin'>('guest');
  const [isClient, setIsClient] = useState(false);
  const router = useRouter();

  // 客户端检测
  useEffect(() => {
    setIsClient(true);
  }, []);

  // 获取用户信息
  useEffect(() => {
    if (isClient) {
      // 检查是否有用户登录信息
      const userInfo = localStorage.getItem('user');
      const savedNickname = localStorage.getItem('nickname');
      
      if (userInfo) {
        try {
          const userData = JSON.parse(userInfo);
          setUser(userData);
          setNickname(userData.nickname);
          // 设置用户角色
          if (userData.username.toLowerCase() === 'admin') {
            setUserRole('admin');
          } else {
            setUserRole('user');
          }
        } catch (error) {
          console.error('解析用户信息失败:', error);
          localStorage.removeItem('user');
        }
      } else if (savedNickname) {
        // 使用昵称快速进入
        if (savedNickname.toLowerCase() === 'admin') {
          message.error('admin昵称已被保留，请使用其他昵称');
          localStorage.removeItem('nickname');
          router.push('/');
          return;
        }
        setNickname(savedNickname);
        setUserRole('guest');
      } else {
        // 没有登录信息，跳转到首页
        router.push('/');
        return;
      }
    }
  }, [isClient, router]);

  // 尝试恢复上次进入的房间
  useEffect(() => {
    if (isClient && nickname) {
      const lastRoomId = localStorage.getItem('lastRoomId');
      if (lastRoomId) {
        setCurrentRoomId(parseInt(lastRoomId));
      }
    }
  }, [isClient, nickname]);

  const handleSelectRoom = (roomId: number) => {
    setCurrentRoomId(roomId);
    if (isClient) {
      localStorage.setItem('lastRoomId', roomId.toString());
    }
  };

  const handleLogout = () => {
    localStorage.removeItem('user');
    localStorage.removeItem('nickname');
    localStorage.removeItem('lastRoomId');
    router.push('/');
  };

  // 在客户端渲染之前显示加载状态
  if (!isClient || !nickname) {
    return (
      <div style={{ 
        display: 'flex', 
        justifyContent: 'center', 
        alignItems: 'center', 
        height: '100vh',
        fontSize: '16px',
        color: '#666'
      }}>
        加载中...
      </div>
    );
  }

  return (
    <Layout className={styles.chatLayout}>
      <Sider width={300} className={styles.sider}>
        <RoomList 
          onSelectRoom={handleSelectRoom}
          currentRoomId={currentRoomId}
          nickname={nickname}
          user={user}
          userRole={userRole}
          onLogout={handleLogout}
        />
      </Sider>
      <Content className={styles.content}>
        <ChatWindow 
          roomId={currentRoomId} 
          user={user}
          userRole={userRole}
        />
      </Content>
    </Layout>
  );
}
