'use client';

import { useState, useEffect } from 'react';
import { Table, Card, Tag, Button, Space, message, Modal } from 'antd';
import { UserOutlined, CrownOutlined, DeleteOutlined } from '@ant-design/icons';
import { User, UserRole, UserStatus } from '@/types/chat';

interface UserWithActions extends User {
  key: string;
}

export default function AdminPage() {
  const [users, setUsers] = useState<UserWithActions[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchUsers();
  }, []);

  const fetchUsers = async () => {
    try {
      setLoading(true);
      // 这里应该调用实际的API，暂时使用模拟数据
      const mockUsers: UserWithActions[] = [
        {
          id: '1',
          username: 'admin',
          email: 'admin@chatroom.com',
          nickname: '系统管理员',
          status: UserStatus.OFFLINE,
          role: UserRole.ADMIN,
          createdAt: new Date(),
          updatedAt: new Date(),
          key: '1',
        },
        {
          id: '2',
          username: 'alice',
          email: 'alice@example.com',
          nickname: '爱丽丝',
          status: UserStatus.ONLINE,
          role: UserRole.USER,
          createdAt: new Date(),
          updatedAt: new Date(),
          key: '2',
        },
        {
          id: '3',
          username: 'bob',
          email: 'bob@example.com',
          nickname: '鲍勃',
          status: UserStatus.OFFLINE,
          role: UserRole.USER,
          createdAt: new Date(),
          updatedAt: new Date(),
          key: '3',
        },
        {
          id: '4',
          username: 'charlie',
          email: 'charlie@example.com',
          nickname: '查理',
          status: UserStatus.AWAY,
          role: UserRole.USER,
          createdAt: new Date(),
          updatedAt: new Date(),
          key: '4',
        },
        {
          id: '5',
          username: 'diana',
          email: 'diana@example.com',
          nickname: '黛安娜',
          status: UserStatus.ONLINE,
          role: UserRole.USER,
          createdAt: new Date(),
          updatedAt: new Date(),
          key: '5',
        },
        {
          id: '6',
          username: 'eve',
          email: 'eve@example.com',
          nickname: '夏娃',
          status: UserStatus.OFFLINE,
          role: UserRole.USER,
          createdAt: new Date(),
          updatedAt: new Date(),
          key: '6',
        },
      ];
      setUsers(mockUsers);
    } catch (error) {
      message.error('获取用户列表失败');
    } finally {
      setLoading(false);
    }
  };

  const getStatusColor = (status: UserStatus) => {
    switch (status) {
      case UserStatus.ONLINE:
        return 'success';
      case UserStatus.AWAY:
        return 'warning';
      case UserStatus.OFFLINE:
        return 'default';
      default:
        return 'default';
    }
  };

  const getStatusText = (status: UserStatus) => {
    switch (status) {
      case UserStatus.ONLINE:
        return '在线';
      case UserStatus.AWAY:
        return '离开';
      case UserStatus.OFFLINE:
        return '离线';
      default:
        return '未知';
    }
  };

  const getRoleColor = (role: UserRole) => {
    return role === UserRole.ADMIN ? 'red' : 'blue';
  };

  const getRoleText = (role: UserRole) => {
    return role === UserRole.ADMIN ? '管理员' : '普通用户';
  };

  const handleDeleteUser = (user: UserWithActions) => {
    if (user.role === UserRole.ADMIN) {
      message.warning('不能删除管理员账户');
      return;
    }

    Modal.confirm({
      title: '确认删除',
      content: `确定要删除用户 "${user.nickname}" 吗？此操作不可撤销。`,
      okText: '确认删除',
      okType: 'danger',
      cancelText: '取消',
      onOk: async () => {
        try {
          // 这里应该调用实际的删除API
          message.success('用户删除成功');
          fetchUsers();
        } catch (error) {
          message.error('删除用户失败');
        }
      },
    });
  };

  const columns = [
    {
      title: '用户信息',
      key: 'userInfo',
      render: (record: UserWithActions) => (
        <Space>
          {record.role === UserRole.ADMIN ? (
            <CrownOutlined style={{ color: '#ff4d4f' }} />
          ) : (
            <UserOutlined />
          )}
          <div>
            <div style={{ fontWeight: 'bold' }}>{record.nickname}</div>
            <div style={{ fontSize: '12px', color: '#666' }}>@{record.username}</div>
          </div>
        </Space>
      ),
    },
    {
      title: '邮箱',
      dataIndex: 'email',
      key: 'email',
    },
    {
      title: '角色',
      dataIndex: 'role',
      key: 'role',
      render: (role: UserRole) => (
        <Tag color={getRoleColor(role)}>
          {getRoleText(role)}
        </Tag>
      ),
    },
    {
      title: '状态',
      dataIndex: 'status',
      key: 'status',
      render: (status: UserStatus) => (
        <Tag color={getStatusColor(status)}>
          {getStatusText(status)}
        </Tag>
      ),
    },
    {
      title: '注册时间',
      dataIndex: 'createdAt',
      key: 'createdAt',
      render: (date: Date) => new Date(date).toLocaleDateString(),
    },
    {
      title: '操作',
      key: 'actions',
      render: (record: UserWithActions) => (
        <Space>
          <Button
            type="link"
            size="small"
            onClick={() => message.info('查看详情功能待实现')}
          >
            查看详情
          </Button>
          {record.role !== UserRole.ADMIN && (
            <Button
              type="link"
              size="small"
              danger
              icon={<DeleteOutlined />}
              onClick={() => handleDeleteUser(record)}
            >
              删除
            </Button>
          )}
        </Space>
      ),
    },
  ];

  return (
    <div style={{ padding: '24px' }}>
      <Card title="用户管理" extra={
        <Button type="primary" onClick={fetchUsers}>
          刷新
        </Button>
      }>
        <Table
          columns={columns}
          dataSource={users}
          loading={loading}
          pagination={{
            pageSize: 10,
            showSizeChanger: true,
            showQuickJumper: true,
            showTotal: (total) => `共 ${total} 个用户`,
          }}
        />
      </Card>

      <Card title="测试账户信息" style={{ marginTop: '24px' }}>
        <div style={{ marginBottom: '16px' }}>
          <h4>👑 管理员账户</h4>
          <p><strong>用户名:</strong> admin</p>
          <p><strong>密码:</strong> admin123</p>
          <p><strong>权限:</strong> 可以删除任何房间和消息，查看所有用户信息</p>
        </div>

        <div style={{ marginBottom: '16px' }}>
          <h4>👤 测试用户账户</h4>
          <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(300px, 1fr))', gap: '16px' }}>
            <div>
              <p><strong>alice / alice123</strong> - 爱丽丝</p>
              <p><strong>bob / bob123</strong> - 鲍勃</p>
              <p><strong>charlie / charlie123</strong> - 查理</p>
            </div>
            <div>
              <p><strong>diana / diana123</strong> - 黛安娜</p>
              <p><strong>eve / eve123</strong> - 夏娃</p>
            </div>
          </div>
        </div>

        <div>
          <h4>🏠 测试房间</h4>
          <ul>
            <li>欢迎大厅 (管理员创建)</li>
            <li>技术讨论 (爱丽丝创建)</li>
            <li>闲聊天地 (鲍勃创建)</li>
            <li>游戏交流 (查理创建)</li>
          </ul>
        </div>
      </Card>
    </div>
  );
}
