const API_BASE_URL = process.env.NEXT_PUBLIC_API_URL;

export const fetcher = async (path: string, args?: RequestInit) => {
  // 使用相对路径 + next.config.ts 的 rewrites 在本地开发同源代理到 3001
  const url = API_BASE_URL ? `${API_BASE_URL}${path}` : path;
  
  try {
    const res = await fetch(url, {
      ...args,
      headers: {
        'Content-Type': 'application/json',
        ...args?.headers,
      },
    });

    // 放行 201/409/401 等业务分支，由下方 json.code 决定是否报错
    if (!res.ok && res.status !== 201 && res.status !== 409 && res.status !== 401) {
      const err: Error & { status?: number } = new Error(`HTTP error! status: ${res.status}`);
      err.status = res.status;
      throw err;
    }

    const json = await res.json();
    
    if (json.code !== 0) {
      const err: Error & { info?: unknown; code?: number } = new Error(json.message || 'API返回错误');
      err.info = json.data;
      err.code = json.code;
      throw err;
    }

    return json.data;
  } catch (error: unknown) {
    // 如果是网络错误
    if (error instanceof Error && error.name === 'TypeError' && error.message.includes('fetch')) {
      throw new Error('网络连接失败，请检查网络连接');
    }

    // 如果是API错误
    if (error instanceof Error && (error as Error & { code?: number }).code !== undefined) {
      throw error;
    }

    // 其他错误
    if (error instanceof Error) {
      throw new Error(error.message || '请求失败');
    }

    throw new Error('未知错误');
  }
};
