'use client';

import React, { useEffect, useRef, useState } from 'react';
import useSWR, { mutate } from 'swr';
import { Spin, Alert, Typography, Empty } from 'antd';
import { fetcher } from '../lib/fetcher';
import { Message, RoomPreviewInfo } from '../types';
import MessageItem from './MessageItem';
import MessageInput from './MessageInput';
import styles from './ChatWindow.module.css';

const { Title } = Typography;

interface ChatWindowProps {
  roomId: number | null;
}

export default function ChatWindow({ roomId }: ChatWindowProps) {
  const messageListRef = useRef<HTMLDivElement>(null);
  const [nickname, setNickname] = useState<string | null>(null);
  const [isClient, setIsClient] = useState(false);

  // 客户端检测
  useEffect(() => {
    setIsClient(true);
  }, []);

  // 获取昵称
  useEffect(() => {
    if (isClient) {
      const storedNickname = localStorage.getItem('nickname');
      setNickname(storedNickname);
    }
  }, [isClient]);

  // 获取房间信息
  const { data: roomData } = useSWR<{ rooms: RoomPreviewInfo[] }>('/api/room/list', fetcher);
  const currentRoom = roomData?.rooms?.find(room => room.roomId === roomId);

  // 获取消息列表
  const {
    data: messageData,
    error,
    isLoading,
  } = useSWR<{ messages: Message[] }>(
    roomId ? `/api/room/message/list?roomId=${roomId}` : null,
    fetcher,
    {
      refreshInterval: 1000, // 每秒轮询一次
    }
  );
  const messages = messageData?.messages || [];

  // 自动滚动到底部
  useEffect(() => {
    if (messageListRef.current) {
      messageListRef.current.scrollTop = messageListRef.current.scrollHeight;
    }
  }, [messages]);

  const handleSendMessage = async (content: string) => {
    if (!roomId || !nickname) return;

    const optimisticNewMessage: Message = {
      messageId: Date.now(), // 临时ID
      roomId,
      sender: nickname,
      content,
      time: Date.now(),
    };

    // 乐观更新UI
    mutate(
      (currentData: any) => ({
        ...currentData,
        messages: [...(currentData?.messages || []), optimisticNewMessage]
      }),
      false
    );

    try {
      await fetcher('/api/message/add', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          roomId,
          sender: nickname,
          content,
        }),
      });
      
      // 触发重新获取以获取服务器端的实际消息
      mutate(`/api/room/message/list?roomId=${roomId}`);
    } catch (e) {
      console.error('发送消息失败:', e);
      // 发送失败时回滚乐观更新
      mutate(
        (currentData: any) => ({
          ...currentData,
          messages: (currentData?.messages || []).filter((msg: Message) => msg.messageId !== optimisticNewMessage.messageId)
        }),
        false
      );
    }
  };

  if (!roomId) {
    return (
      <div className={styles.placeholder}>
        <Empty 
          description="请选择一个聊天房间开始聊天"
          image={Empty.PRESENTED_IMAGE_SIMPLE}
        />
      </div>
    );
  }

  if (isLoading && !messages.length) {
    return (
      <div className={styles.placeholder}>
        <Spin size="large" />
      </div>
    );
  }

  if (error) {
    return (
      <div className={styles.placeholder}>
        <Alert 
          message="错误" 
          description="加载消息失败，请稍后重试" 
          type="error" 
          showIcon 
        />
      </div>
    );
  }

  return (
    <div className={styles.chatWindow}>
      {/* 房间标题 */}
      <div className={styles.roomHeader}>
        <Title level={4} className={styles.roomTitle}>
          {currentRoom?.roomName || `房间 ${roomId}`}
        </Title>
      </div>

      {/* 消息列表 */}
      <div ref={messageListRef} className={styles.messageList}>
        {messages.length === 0 ? (
          <div className={styles.emptyMessages}>
            <Empty 
              description="暂无消息，开始聊天吧！"
              image={Empty.PRESENTED_IMAGE_SIMPLE}
            />
          </div>
        ) : (
          messages.map((msg) => (
            <MessageItem
              key={msg.messageId}
              message={msg}
              isOwnMessage={msg.sender === nickname}
            />
          ))
        )}
      </div>

      {/* 消息输入框 */}
      <footer className={styles.chatFooter}>
        <MessageInput onSendMessage={handleSendMessage} />
      </footer>
    </div>
  );
}
