package controller

import (
	"go-svc-tpl/app/response"
	"go-svc-tpl/model"
	"go-svc-tpl/utils"
	"strconv"

	"github.com/labstack/echo/v4"
	"github.com/sirupsen/logrus"
)

func Register(c echo.Context) error {
	name := c.FormValue("name")
	email := c.FormValue("email")
	pwd := c.FormValue("pwd")

	// !!! without validation now
	err := model.RegisterUser(name, email, pwd)

	if err != nil {
		logrus.Error(err)
		return response.SendResponse(c, 1, "register failed", "")
	}
	return response.SendResponse(c, 0, "register successfully", "")
}

// get token from (email, pwd)
func Login(c echo.Context) error {
	email := c.FormValue("email")
	pwd := c.FormValue("pwd")

	data := make(map[string]interface{})
	var msg string
	var code int

	// !!! without validation now
	// judge whether the user exists
	exist := model.CheckAuth(email, pwd)
	if exist {
		token, err := utils.GenToken(email, pwd)
		if err != nil {
			code = 1
			msg = "can't generate token"
			logrus.Error(err)
		} else {
			code = 0
			msg = "generate token successfully"
			data["token"] = token
		}
	} else {
		code = 1
		msg = "user not exist"
	}
	return response.SendResponse(c, code, msg, data)
}

// by frontend ?
func Logout(c echo.Context) error {
	return nil
}

func Info(c echo.Context) error {
	name := getName(c)
	user, err := model.GetUser(name)
	if err != nil {
		logrus.Error(err)
		return response.SendResponse(c, 1, "get user failed", "")
	}

	data := struct {
		Name  string `json:"name"`
		Email string `json:"Email"`
	}{name, user.Email}

	return response.SendResponse(c, 0, "get info successfully", data)
}

func GetRecord(c echo.Context) error {
	name := getName(c)
	data, err := model.GetRecord(name)
	if err != nil {
		logrus.Error(err)
		return response.SendResponse(c, 1, "get failed", "")
	}

	return response.SendResponse(c, 0, "get record successfully", data)
}

func GetUrl(c echo.Context) error {
	name := getName(c)
	data, err := model.GetUrl(name)
	if err != nil {
		logrus.Error(err)
		return response.SendResponse(c, 1, "get failed", "")
	}

	return response.SendResponse(c, 0, "get url successfully", data)
}

func Create(c echo.Context) error {
	// get data of shortlink from request
	var link model.ShortLink
	if err := c.Bind(&link); err != nil {
		logrus.Error(err)
		return response.SendResponse(c, 1, "request error", "")
	}

	if link.Short == "" {
		link.Short = utils.GenShort()
	}

	// create a new link to user by name
	name := getName(c)
	var id int
	var err error

	if id, err = model.CreateLink(name, link); err != nil {
		logrus.Error(err)
		return response.SendResponse(c, 1, "create failed", "")
	}
	return response.SendResponse(c, 0, "create successfully", struct {
		ID int `json:"id"`
	}{id})
}

func Query(c echo.Context) error {
	name := getName(c)
	id, _ := strconv.Atoi(c.FormValue("id"))
	var link model.ShortLink
	var err error

	if link, err = model.QueryLink(name, id); err != nil {
		logrus.Error(err)
		return response.SendResponse(c, 1, "query error", "")
	}
	return response.SendResponse(c, 0, "query successfully", link)
}

func Update(c echo.Context) error {
	// read update data
	var data model.UpdateData
	if err := c.Bind(&data); err != nil {
		logrus.Error(err)
		return response.SendResponse(c, 1, "request error", "")
	}

	// update the data of user
	name := getName(c)
	if err := model.UpdateLink(name, data); err != nil {
		logrus.Error(err)
		return response.SendResponse(c, 1, "update error", "")
	}

	return response.SendResponse(c, 0, "update successfully", "")
}

func Delete(c echo.Context) error {
	name := getName(c)
	id, _ := strconv.Atoi(c.FormValue("id"))

	if err := model.DeleteLink(name, id); err != nil {
		logrus.Error(err)
		return response.SendResponse(c, 1, "delete error", "")
	}
	return response.SendResponse(c, 0, "delete successfully", "")
}

func Pause(c echo.Context) error {
	name := getName(c)
	id, _ := strconv.Atoi(c.FormValue("id"))

	if err := model.PauseLink(name, id); err != nil {
		logrus.Error(err)
		return response.SendResponse(c, 1, "puase error", "")
	}
	return response.SendResponse(c, 0, "pause succesfully", "")
}

// get the username by token
func getName(c echo.Context) string {
	token := c.FormValue("token")
	claims, _ := utils.ParseToken(token)
	return claims.Username
}
