// 文件位置：src/lib/dataStore.ts
// 简单的内存数据存储，用于模拟数据库
// 实际项目中应该使用真正的数据库（因为实验文档使用数据库的要求放在后面，我写完才看见，所以只能维持之前用内存存储的方法了）
import { RoomPreviewInfo, Message } from "@/types";

// 声明全局变量类型
declare global {
  var __chatroom_messages__: Message[] | undefined;
  var __chatroom_rooms__: RoomPreviewInfo[] | undefined;
  var __chatroom_users__: UserData[] | undefined;
  var __chatroom_next_user_id__: number | undefined;
}

// 初始房间数据
const initialRooms: RoomPreviewInfo[] = [
  {
    roomId: 1,
    roomName: "技术交流",
    lastMessage: {
      messageId: 1,
      roomId: 1,
      sender: "张三",
      content: "大家好！",
      time: Date.now() - 300000,
    },
  },
  {
    roomId: 2,
    roomName: "项目讨论",
    lastMessage: {
      messageId: 2,
      roomId: 2,
      sender: "李四",
      content: "会议改时间了",
      time: Date.now() - 600000,
    },
  },
];

// 初始消息数据
const initialMessages: Message[] = [
  {
    messageId: 1,
    roomId: 1,
    sender: "张三",
    content: "大家好！",
    time: Date.now() - 3600000,
  },
  {
    messageId: 2,
    roomId: 1,
    sender: "李四",
    content: "你好呀！",
    time: Date.now() - 1800000,
  },
  {
    messageId: 3,
    roomId: 2,
    sender: "王五",
    content: "项目进度如何？",
    time: Date.now() - 2400000,
  },
  {
    messageId: 4,
    roomId: 2,
    sender: "李四",
    content: "会议改时间了",
    time: Date.now() - 600000,
  },
];

// 初始化全局数据
if (typeof global !== "undefined") {
  // 初始化房间数据
  if (!global.__chatroom_rooms__) {
    global.__chatroom_rooms__ = [...initialRooms];
    console.log("[全局初始化] 初始化房间数据");
  }

  // 初始化消息数据
  if (!global.__chatroom_messages__) {
    global.__chatroom_messages__ = [...initialMessages];
    console.log("[全局初始化] 初始化消息数据");
  }

  // 初始化用户数据
  if (!global.__chatroom_users__) {
    global.__chatroom_users__ = [];
    console.log("[全局初始化] 初始化用户数据");
  }

  if (!global.__chatroom_next_user_id__) {
    global.__chatroom_next_user_id__ = 1;
    console.log("[全局初始化] 初始化用户ID计数器");
  }
}

// 房间数据
export let rooms: RoomPreviewInfo[] = [];

// 消息数据
export let messages: Message[] = [];

// 用户数据
let users: UserData[] = [];
let nextUserId = 1;

// 在模块加载时同步全局数据
if (typeof global !== "undefined") {
  // 同步房间数据
  if (global.__chatroom_rooms__) {
    rooms = global.__chatroom_rooms__;
    console.log("[模块加载] 已从全局变量加载房间数据，房间数:", rooms.length);
  } else {
    rooms = [...initialRooms];
    console.log("[模块加载] 使用初始房间数据，房间数:", rooms.length);
  }

  // 同步消息数据
  if (global.__chatroom_messages__) {
    messages = global.__chatroom_messages__;
    console.log(
      "[模块加载] 已从全局变量加载消息数据，消息数:",
      messages.length
    );
  } else {
    messages = [...initialMessages];
    console.log("[模块加载] 使用初始消息数据，消息数:", messages.length);
  }

  // 同步用户数据
  if (global.__chatroom_users__) {
    users = global.__chatroom_users__;
    console.log("[模块加载] 已从全局变量加载用户数据，用户数:", users.length);
  }

  if (global.__chatroom_next_user_id__) {
    nextUserId = global.__chatroom_next_user_id__;
    console.log("[模块加载] 已从全局变量加载用户ID计数器:", nextUserId);
  }
} else {
  rooms = [...initialRooms];
  messages = [...initialMessages];
  console.log("[模块加载] global不可用，使用初始数据");
}

// 同步房间数据到全局变量
function syncRoomsToGlobal(): void {
  if (typeof global !== "undefined") {
    global.__chatroom_rooms__ = rooms;
    console.log(
      "[syncRoomsToGlobal] 房间数据已同步到全局变量，房间数:",
      rooms.length
    );
  }
}

// 同步消息数据到全局变量
function syncMessagesToGlobal(): void {
  if (typeof global !== "undefined") {
    global.__chatroom_messages__ = messages;
    console.log(
      "[syncMessagesToGlobal] 消息数据已同步到全局变量，消息数:",
      messages.length
    );
  }
}

// 获取下一个房间ID
export function getNextRoomId(): number {
  // 确保从全局变量同步最新数据
  if (typeof global !== "undefined" && global.__chatroom_rooms__) {
    rooms = global.__chatroom_rooms__;
  }

  const nextId = Math.max(...rooms.map((r) => r.roomId), 0) + 1;
  console.log(`[getNextRoomId] 生成的下一个房间ID: ${nextId}`);
  return nextId;
}

// 获取下一个消息ID
export function getNextMessageId(): number {
  // 确保从全局变量同步最新数据
  if (typeof global !== "undefined" && global.__chatroom_messages__) {
    messages = global.__chatroom_messages__;
  }

  const nextId = Math.max(...messages.map((m) => m.messageId), 0) + 1;
  console.log(
    `[getNextMessageId] 当前消息数量: ${messages.length}, 生成的下一个ID: ${nextId}`
  );
  return nextId;
}

// 添加房间
export function addRoom(room: RoomPreviewInfo): void {
  console.log(`\n[addRoom] ===== 开始添加房间 =====`);
  console.log(`[addRoom] 要添加的房间:`, {
    roomId: room.roomId,
    roomName: room.roomName,
  });

  // 确保从全局变量同步最新数据
  if (typeof global !== "undefined" && global.__chatroom_rooms__) {
    rooms = global.__chatroom_rooms__;
  }

  console.log(`[addRoom] 添加前房间数量: ${rooms.length}`);
  rooms.push(room);

  // 同步到全局变量
  syncRoomsToGlobal();

  console.log(`[addRoom] 添加后房间数量: ${rooms.length}`);
  console.log(
    `[addRoom] 所有房间:`,
    rooms.map((r) => `${r.roomName}(ID:${r.roomId})`)
  );
  console.log(`[addRoom] ===== 房间添加完成 =====\n`);
}

// 删除房间
export function removeRoom(roomId: number): boolean {
  console.log(`\n[removeRoom] ===== 开始删除房间 =====`);
  console.log(`[removeRoom] 要删除的房间ID: ${roomId}`);

  // 确保从全局变量同步最新数据
  if (typeof global !== "undefined") {
    if (global.__chatroom_rooms__) {
      rooms = global.__chatroom_rooms__;
    }
    if (global.__chatroom_messages__) {
      messages = global.__chatroom_messages__;
    }
  }

  console.log(`[removeRoom] 删除前房间数量: ${rooms.length}`);
  console.log(`[removeRoom] 删除前消息数量: ${messages.length}`);

  const index = rooms.findIndex((r) => r.roomId === roomId);
  if (index !== -1) {
    rooms.splice(index, 1);

    // 同时删除该房间的所有消息
    const originalMessageCount = messages.length;
    messages = messages.filter((m) => m.roomId !== roomId);
    const deletedMessageCount = originalMessageCount - messages.length;

    // 同步到全局变量
    syncRoomsToGlobal();
    syncMessagesToGlobal();

    console.log(`[removeRoom] 房间删除成功`);
    console.log(`[removeRoom] 删除后房间数量: ${rooms.length}`);
    console.log(`[removeRoom] 删除了 ${deletedMessageCount} 条消息`);
    console.log(`[removeRoom] 删除后消息数量: ${messages.length}`);
    console.log(
      `[removeRoom] 剩余房间:`,
      rooms.map((r) => `${r.roomName}(ID:${r.roomId})`)
    );
    console.log(`[removeRoom] ===== 房间删除完成 =====\n`);
    return true;
  }

  console.log(`[removeRoom] 房间不存在，删除失败`);
  console.log(`[removeRoom] ===== 房间删除失败 =====\n`);
  return false;
}

// 添加消息
export function addMessage(message: Message): void {
  console.log(`\n[addMessage] ===== 开始添加消息 =====`);
  console.log(`[addMessage] 传入的消息:`, {
    messageId: message.messageId,
    roomId: message.roomId,
    sender: message.sender,
    content: message.content,
    time: new Date(message.time).toLocaleString(),
  });

  // 确保从全局变量同步最新数据
  if (typeof global !== "undefined") {
    if (global.__chatroom_messages__) {
      messages = global.__chatroom_messages__;
    }
    if (global.__chatroom_rooms__) {
      rooms = global.__chatroom_rooms__;
    }
  }

  console.log(`[addMessage] 添加前消息总数: ${messages.length}`);

  messages.push(message);

  // 同步消息到全局变量
  syncMessagesToGlobal();

  console.log(`[addMessage] 添加后消息总数: ${messages.length}`);

  // 更新对应房间的最后一条消息
  const roomIndex = rooms.findIndex((r) => r.roomId === message.roomId);
  if (roomIndex !== -1) {
    rooms[roomIndex].lastMessage = message;
    // 同步房间数据到全局变量
    syncRoomsToGlobal();
    console.log(`[addMessage] 房间 ${message.roomId} 的最后消息已更新`);
  } else {
    console.log(`[addMessage] 警告: 未找到房间 ${message.roomId}`);
  }
  console.log(`[addMessage] ===== 消息添加完成 =====\n`);
}

// 获取房间消息
export function getRoomMessages(roomId: number): Message[] {
  console.log(`\n[getRoomMessages] ===== 开始获取房间消息 =====`);
  console.log(`[getRoomMessages] 请求的房间ID: ${roomId}`);

  // 确保从全局变量同步最新数据
  if (typeof global !== "undefined" && global.__chatroom_messages__) {
    messages = global.__chatroom_messages__;
    console.log(`[getRoomMessages] 已从全局变量同步消息数据`);
  }

  console.log(`[getRoomMessages] 当前全部消息数量: ${messages.length}`);

  const roomMessages = messages.filter((m) => m.roomId === roomId);
  console.log(
    `[getRoomMessages] 找到房间 ${roomId} 的消息数量: ${roomMessages.length}`
  );
  console.log(`[getRoomMessages] ===== 房间消息获取完成 =====\n`);

  return roomMessages;
}

// 获取房间更新消息
export function getRoomUpdateMessages(
  roomId: number,
  sinceMessageId: number
): Message[] {
  console.log(
    `[getRoomUpdateMessages] 获取房间 ${roomId} 自消息ID ${sinceMessageId} 以来的更新`
  );

  // 确保从全局变量同步最新数据
  if (typeof global !== "undefined" && global.__chatroom_messages__) {
    messages = global.__chatroom_messages__;
  }

  const updateMessages = messages.filter(
    (m) => m.roomId === roomId && m.messageId > sinceMessageId
  );
  console.log(
    `[getRoomUpdateMessages] 找到 ${updateMessages.length} 条更新消息`
  );
  return updateMessages;
}

// 获取所有房间（用于API）
export function getAllRooms(): RoomPreviewInfo[] {
  // 确保从全局变量同步最新数据
  if (typeof global !== "undefined" && global.__chatroom_rooms__) {
    rooms = global.__chatroom_rooms__;
  }

  console.log(`[getAllRooms] 返回 ${rooms.length} 个房间`);
  return rooms;
}

// 登录鉴权
interface UserData {
  userId: number;
  username: string;
  email?: string;
  passwordHash: string;
  createdAt: number;
}

// 保存用户数据到全局变量
function saveUsersToGlobal(): void {
  if (typeof global !== "undefined") {
    global.__chatroom_users__ = users;
    global.__chatroom_next_user_id__ = nextUserId;
    console.log(
      "[saveUsersToGlobal] 用户数据已保存到全局变量，用户数:",
      users.length
    );
  }
}

// 用户相关操作
export const userStore = {
  // 创建用户
  createUser: async (
    username: string,
    email: string | undefined,
    passwordHash: string
  ): Promise<UserData> => {
    // 确保从全局变量同步最新数据
    if (typeof global !== "undefined") {
      if (global.__chatroom_users__) {
        users = global.__chatroom_users__;
      }
      if (global.__chatroom_next_user_id__) {
        nextUserId = global.__chatroom_next_user_id__;
      }
    }

    const user: UserData = {
      userId: nextUserId++,
      username,
      email,
      passwordHash,
      createdAt: Date.now(),
    };
    users.push(user);

    // 保存到全局变量
    saveUsersToGlobal();

    console.log(`用户创建成功: ${username}, ID: ${user.userId}`);
    console.log(`当前用户总数: ${users.length}`);

    return user;
  },

  // 根据用户名查找用户
  findUserByUsername: (username: string): UserData | undefined => {
    // 确保从全局变量同步最新数据
    if (typeof global !== "undefined" && global.__chatroom_users__) {
      users = global.__chatroom_users__;
    }

    const user = users.find((user) => user.username === username);
    console.log(`查找用户 "${username}":`, user ? "找到" : "未找到");

    return user;
  },

  // 根据用户ID查找用户
  findUserById: (userId: number): UserData | undefined => {
    // 确保从全局变量同步最新数据
    if (typeof global !== "undefined" && global.__chatroom_users__) {
      users = global.__chatroom_users__;
    }
    return users.find((user) => user.userId === userId);
  },

  // 检查用户名是否已存在
  isUsernameExists: (username: string): boolean => {
    // 确保从全局变量同步最新数据
    if (typeof global !== "undefined" && global.__chatroom_users__) {
      users = global.__chatroom_users__;
    }

    const exists = users.some((user) => user.username === username);
    console.log(
      `检查用户名 "${username}" 是否存在:`,
      exists ? "存在" : "不存在"
    );
    return exists;
  },

  // 获取所有用户 (调试用)
  getAllUsers: (): UserData[] => {
    // 确保从全局变量同步最新数据
    if (typeof global !== "undefined" && global.__chatroom_users__) {
      users = global.__chatroom_users__;
    }

    console.log(
      "当前所有用户:",
      users.map((u) => ({
        userId: u.userId,
        username: u.username,
        createdAt: new Date(u.createdAt).toLocaleString(),
      }))
    );

    return users.map(({ passwordHash, ...user }) => user as any);
  },
};
