"use client";

import React, { useState } from "react";
// 引入 useAuth 自定义钩子，用于处理登录相关的逻辑
// useAuth 是一个自定义钩子，封装了认证逻辑，遵循 React 钩子命名规范（以 use 开头）
import { useAuth } from "@/hooks/useAuth";
import "./Auth.css";

// 定义 LoginFormProps 接口，描述组件接收的属性
interface LoginFormProps {
  onSuccess?: () => void;
  switchToRegister: () => void; // 切换到注册模式的函数
}

// 定义 LoginForm 组件，接收 onSuccess 和 switchToRegister 属性
export default function LoginForm({
  onSuccess,
  switchToRegister,
}: LoginFormProps) {
  // 定义状态变量：用户名、密码、加载状态和错误信息
  const [username, setUsername] = useState("");
  const [password, setPassword] = useState("");
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState("");

  // 从 useAuth 钩子中获取 login 函数，用于处理登录
  const { login } = useAuth();

  // 定义 handleSubmit 函数，处理表单提交
  // 当用户点击“登录”按钮时，这个函数会被调用，负责检查输入、发送登录请求。
  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    // 检查用户名和密码是否为空
    if (!username.trim() || !password.trim()) {
      setError("请输入用户名和密码");
      return;
    }

    setLoading(true);
    setError("");

    try {
      // 发送 POST 请求到登录 API
      // 使用 fetch 函数发送 HTTP POST 请求到 /api/auth/login 接口，携带用户名和密码。
      const response = await fetch("/api/auth/login", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({ username, password }),
      });

      // 解析服务器返回的 JSON 数据
      const data = await response.json();

      // 检查服务器返回的数据：如果 data.code 是 0 且 data.data 存在，说明登录成功
      if (data.code === 0 && data.data) {
        login(data.data);
        onSuccess?.();
      } else {
        setError(data.message || "登录失败");
      }
    } catch (error) {
      setError("网络错误，请重试");
      console.error("Login error:", error);
    } finally {
      setLoading(false);
    }
  };

  return (
    <form onSubmit={handleSubmit} className="auth-form">
      <h2>用户登录</h2>

      {error && <div className="auth-error">{error}</div>}

      <div className="auth-field">
        <label htmlFor="username">用户名</label>
        <input
          id="username"
          type="text"
          value={username}
          onChange={(e) => setUsername(e.target.value)}
          placeholder="请输入用户名"
          disabled={loading}
        />
      </div>

      <div className="auth-field">
        <label htmlFor="password">密码</label>
        <input
          id="password"
          type="password"
          value={password}
          onChange={(e) => setPassword(e.target.value)}
          placeholder="请输入密码"
          disabled={loading}
        />
      </div>

      <button type="submit" className="auth-button" disabled={loading}>
        {loading ? "登录中..." : "登录"}
      </button>

      <div className="auth-switch">
        还没有账户？
        <button type="button" onClick={switchToRegister} className="auth-link">
          立即注册
        </button>
      </div>
    </form>
  );
}
