// 这段代码实现了一个用于管理用户认证状态的 AuthStore 类，它通过 localStorage 持久化存储认证信息，并提供了登录、登出、状态订阅等功能
"use client";

import { AuthResponse } from "@/types/auth";

const TOKEN_KEY = "chatroom_token";
const USER_KEY = "chatroom_user";

// AuthState 是认证状态的接口定义
export interface AuthState {
  isAuthenticated: boolean; //表示用户是否已认证。
  user: {
    userId: number;
    username: string;
    email?: string;
  } | null;
  token: string | null;
}

class AuthStore {
  //state 保存了当前的认证状态，初始值表示用户未认证，user 和 token 都是 null。
  private state: AuthState = {
    isAuthenticated: false,
    user: null,
    token: null,
  };

  // listeners 是一个函数数组，每个函数都会在认证状态发生变化时被调用，用于通知状态变化
  private listeners: ((state: AuthState) => void)[] = [];

  constructor() {
    // 从localStorage恢复状态
    if (typeof window !== "undefined") {
      this.loadFromStorage();
    }
  }

  // 订阅状态变化
  subscribe(listener: (state: AuthState) => void) {
    this.listeners.push(listener);
    return () => {
      this.listeners = this.listeners.filter((l) => l !== listener);
    };
  }

  // 通知所有订阅者
  private notify() {
    this.listeners.forEach((listener) => listener(this.state));
  }

  // 获取当前状态
  getState(): AuthState {
    return { ...this.state };
  }

  // 登录
  login(authResponse: AuthResponse) {
    this.state = {
      isAuthenticated: true,
      user: authResponse.user,
      token: authResponse.token,
    };
    this.saveToStorage();
    this.notify();
  }

  // 登出
  logout() {
    this.state = {
      isAuthenticated: false,
      user: null,
      token: null,
    };
    this.clearStorage();
    this.notify();
  }

  // 获取认证头
  getAuthHeader(): Record<string, string> {
    if (!this.state.token) {
      return {};
    }
    return {
      Authorization: `Bearer ${this.state.token}`,
    };
  }

  // 保存到本地存储
  // saveToStorage 方法将当前的认证信息（token 和 user）保存到 localStorage 中。在保存之前，首先检查是否在浏览器环境中。
  private saveToStorage() {
    if (typeof window === "undefined") return;

    localStorage.setItem(TOKEN_KEY, this.state.token || "");
    localStorage.setItem(USER_KEY, JSON.stringify(this.state.user));
  }

  // 从本地存储加载
  // loadFromStorage 方法从 localStorage 中获取保存的认证信息。如果存在有效的 token 和用户信息，它会将这些信息加载到 state 中。若加载过程中出错（例如数据损坏），则清除存储。
  private loadFromStorage() {
    if (typeof window === "undefined") return;

    const token = localStorage.getItem(TOKEN_KEY);
    const userStr = localStorage.getItem(USER_KEY);

    if (token && userStr) {
      try {
        const user = JSON.parse(userStr);
        this.state = {
          isAuthenticated: true,
          user,
          token,
        };
      } catch (error) {
        console.error("Failed to load auth state from storage:", error);
        this.clearStorage();
      }
    }
  }

  // 清除本地存储
  private clearStorage() {
    if (typeof window === "undefined") return;

    localStorage.removeItem(TOKEN_KEY);
    localStorage.removeItem(USER_KEY);
  }
}

export const authStore = new AuthStore();
