// 文件位置：src/lib/api.ts
// 管理认证 token（从 localStorage 获取，设置 Authorization 头）。
// 定义通用 fetcher 函数（getFetcher、postFetcher、getMessageUpdateFetcher），处理 GET/POST 请求、错误和 JSON 解析。
// 封装具体 API 接口（api 对象），如获取/创建/删除房间、发送/获取消息，支持认证和类型安全。

// 引入 API 响应类型，用于定义各种接口返回数据的结构
import {
  RoomListRes, //（房间列表响应）
  RoomMessageListRes, //（房间消息列表响应）
  RoomAddRes, //（创建房间响应）
  RoomMessageGetUpdateRes, //（消息更新响应）
  ApiResponse, //（通用 API 响应类型）
} from "@/types/api";

// 认证状态管理 - 简单版本，避免循环依赖
// authToken 像一个“临时身份证”，用来证明用户已登录
let authToken: string | null = null;

// 设置认证 token
// 定义了一个导出函数 setAuthToken，接收一个 token 参数（字符串或 null），将它赋值给全局的 authToken。
export function setAuthToken(token: string | null) {
  authToken = token;
}

// 获取认证头
function getAuthHeaders(): Record<string, string> {
  const headers: Record<string, string> = {
    "Content-Type": "application/json",
  };

  // 从 localStorage 获取 token（客户端）
  if (typeof window !== "undefined") {
    const token = localStorage.getItem("chatroom_token");
    if (token) {
      headers["Authorization"] = `Bearer ${token}`;
    }
  }

  return headers;
}

// GET请求的fetcher函数
// 定义了一个导出的常量函数 getFetcher，用于发送 GET 请求（获取数据），接收 url 参数（字符串）。使用 fetch API 发送请求，传入 getAuthHeaders() 生成的头。
export const getFetcher = async (url: string) => {
  const response = await fetch(url, {
    headers: getAuthHeaders(),
  });

  if (!response.ok) {
    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
  }

  const result: ApiResponse = await response.json();

  if (result.code !== 0) {
    throw new Error(result.message || "请求失败");
  }

  return result.data;
};

// POST请求的fetcher函数
export const postFetcher = async (url: string, { arg }: { arg: any }) => {
  const response = await fetch(url, {
    method: "POST",
    headers: getAuthHeaders(),
    body: JSON.stringify(arg),
  });

  if (!response.ok) {
    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
  }

  const result: ApiResponse = await response.json();

  if (result.code !== 0) {
    throw new Error(result.message || "请求失败");
  }

  return result.data;
};

// 专门用于获取消息更新的fetcher
export const getMessageUpdateFetcher = async (url: string) => {
  const response = await fetch(url, {
    headers: getAuthHeaders(),
  });

  if (!response.ok) {
    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
  }

  const result: ApiResponse<RoomMessageGetUpdateRes> = await response.json();

  if (result.code !== 0) {
    throw new Error(result.message || "获取消息更新失败");
  }

  // 确保数据存在，否则抛出错误
  if (!result.data) {
    throw new Error("服务器返回的数据为空");
  }

  return result.data;
};

// API接口封装函数 - 更新为使用认证
export const api = {
  // 获取房间列表
  getRoomList: async (): Promise<RoomListRes> => {
    return getFetcher("/api/room/list");
  },

  // 创建房间 - 不再需要传递 user 参数
  createRoom: async (roomName: string): Promise<RoomAddRes> => {
    return postFetcher("/api/room/add", { arg: { roomName } });
  },

  // 删除房间 - 不再需要传递 user 参数
  deleteRoom: async (roomId: number): Promise<void> => {
    return postFetcher("/api/room/delete", { arg: { roomId } });
  },

  // 获取房间消息列表
  getRoomMessages: async (roomId: number): Promise<RoomMessageListRes> => {
    return getFetcher(`/api/room/message/list?roomId=${roomId}`);
  },

  // 发送消息 - 不再需要传递 sender 参数
  sendMessage: async (roomId: number, content: string): Promise<void> => {
    return postFetcher("/api/message/add", { arg: { roomId, content } });
  },

  // 获取消息更新
  getMessageUpdate: async (
    roomId: number,
    sinceMessageId: number
  ): Promise<RoomMessageGetUpdateRes> => {
    return getMessageUpdateFetcher(
      `/api/room/message/getUpdate?roomId=${roomId}&sinceMessageId=${sinceMessageId}`
    );
  },
};
