// 文件位置：src/components/ChatRoom/ChatRoom.tsx
"use client";
import { useState, useEffect, useRef } from "react";
import useSWR from "swr";
import useSWRMutation from "swr/mutation";
import { Message, RoomPreviewInfo, ChatRoomData } from "@/types";
import {
  RoomListRes,
  RoomMessageListRes,
  RoomAddRes,
  RoomMessageGetUpdateRes,
} from "@/types/api";
import { getFetcher, postFetcher } from "@/lib/api";
import RoomEntry from "@/components/RoomEntry/RoomEntry";
import MessageItem from "@/components/MessageItem/MessageItem";
import "./ChatRoom.css";

export default function ChatRoom() {
  // 从localStorage获取用户名，如果没有则默认为"我"
  const [currentUser] = useState(() => {
    if (typeof window !== "undefined") {
      return localStorage.getItem("userName") || "我";
    }
    return "我";
  });

  const [message, setMessage] = useState("");
  const [roomId, setRoomId] = useState<number | null>(null);
  const [newRoomName, setNewRoomName] = useState("");
  const [showCreateForm, setShowCreateForm] = useState(false);

  // 添加一个标记来跟踪是否已经初始化了房间选择
  const [hasInitializedRoom, setHasInitializedRoom] = useState(false);

  // 用于跟踪最后一条消息ID，实现增量更新
  const lastMessageIdRef = useRef<number>(0);

  // 右键菜单状态
  const [contextMenu, setContextMenu] = useState<{
    visible: boolean;
    x: number;
    y: number;
    roomId: number | null;
  }>({
    visible: false,
    x: 0,
    y: 0,
    roomId: null,
  });

  // 获取房间列表
  const {
    data: roomListData,
    error: roomError,
    isLoading: roomIsLoading,
    mutate: mutateRooms,
  } = useSWR<RoomListRes>("/api/room/list", getFetcher, {
    refreshInterval: 5000,
    revalidateOnFocus: false,
    revalidateOnReconnect: true,
  });

  // 获取当前房间的消息列表
  const {
    data: messageListData,
    error: messageListError,
    isLoading: messageListIsLoading,
    mutate: mutateMessages,
  } = useSWR<RoomMessageListRes>(
    () => {
      if (roomId === null) return null;
      return `/api/room/message/list?roomId=${roomId}`;
    },
    getFetcher,
    {
      refreshInterval: 3000,
      revalidateOnFocus: false,
      onSuccess: (data) => {
        if (data.messages.length > 0) {
          const maxId = Math.max(...data.messages.map((m) => m.messageId));
          lastMessageIdRef.current = maxId;
        }
      },
    }
  );

  // 创建房间的mutation - 更新类型定义
  const { trigger: createRoomTrigger, isMutating: isCreatingRoom } =
    useSWRMutation<RoomAddRes, Error, string, { roomName: string }>(
      "/api/room/add",
      postFetcher
    );

  // 发送消息的mutation - 更新类型定义
  const { trigger: sendMessageTrigger, isMutating: isSendingMessage } =
    useSWRMutation<void, Error, string, { roomId: number; content: string }>(
      "/api/message/add",
      postFetcher
    );

  // 删除房间的mutation - 更新类型定义
  const { trigger: deleteRoomTrigger, isMutating: isDeletingRoom } =
    useSWRMutation<void, Error, string, { roomId: number }>(
      "/api/room/delete",
      postFetcher
    );

  // 自动选择房间的逻辑 - 修改为确保始终有房间被选中
  useEffect(() => {
    if (typeof window === "undefined") return;
    if (!roomListData?.rooms || roomListData.rooms.length === 0) return;

    // 如果已经有选中的房间且房间仍然存在，不重复设置
    if (roomId !== null) {
      const roomExists = roomListData.rooms.some(
        (room) => room.roomId === roomId
      );
      if (roomExists) return;
    }

    // 尝试从localStorage恢复上次选择的房间
    const savedRoomId = localStorage.getItem("lastSelectedRoomId");
    if (savedRoomId && !hasInitializedRoom) {
      const savedId = parseInt(savedRoomId);
      const roomExists = roomListData.rooms.some(
        (room) => room.roomId === savedId
      );
      if (roomExists) {
        setRoomId(savedId);
        setHasInitializedRoom(true);
        return;
      }
    }

    // 如果没有保存的房间ID或房间不存在，选择第一个房间
    console.log("自动选择第一个房间:", roomListData.rooms[0].roomId);
    setRoomId(roomListData.rooms[0].roomId);
    setHasInitializedRoom(true);
  }, [roomListData, hasInitializedRoom]); // 移除roomId依赖，添加hasInitializedRoom

  // 保存房间选择到localStorage
  useEffect(() => {
    if (typeof window !== "undefined" && roomId !== null) {
      localStorage.setItem("lastSelectedRoomId", roomId.toString());
    }
  }, [roomId]);

  // 处理房间点击
  const handleRoomClick = (newRoomId: number) => {
    setRoomId(newRoomId);
    lastMessageIdRef.current = 0;
  };

  // 发送消息 - 移除sender字段
  const sendMessage = async () => {
    const messageContent = message.trim();
    if (!messageContent || !roomId || isSendingMessage) {
      return;
    }

    console.log("准备发送消息:", {
      roomId,
      content: messageContent,
    });

    try {
      await sendMessageTrigger({
        roomId,
        content: messageContent,
      });

      console.log("消息发送成功");
      setMessage("");

      // 立即刷新消息列表
      mutateMessages();
    } catch (error) {
      console.error("发送消息失败:", error);
      alert(
        `发送消息失败: ${error instanceof Error ? error.message : "未知错误"}`
      );
    }
  };

  // 创建新房间 - 移除user字段
  const createNewRoom = async () => {
    if (newRoomName.trim() && !isCreatingRoom) {
      try {
        const result = await createRoomTrigger({
          roomName: newRoomName.trim(),
        });

        console.log("房间创建成功:", result);

        setNewRoomName("");
        setShowCreateForm(false);

        // 刷新房间列表
        await mutateRooms();

        // 自动切换到新建的房间
        setRoomId(result.roomId);
      } catch (error) {
        console.error("创建房间失败:", error);
        alert(
          `创建房间失败: ${error instanceof Error ? error.message : "未知错误"}`
        );
      }
    }
  };

  // 处理右键菜单
  const handleRoomContextMenu = (e: React.MouseEvent, roomId: number) => {
    e.preventDefault();
    setContextMenu({
      visible: true,
      x: e.clientX,
      y: e.clientY,
      roomId: roomId,
    });
  };

  // 关闭右键菜单
  const closeContextMenu = () => {
    setContextMenu({
      visible: false,
      x: 0,
      y: 0,
      roomId: null,
    });
  };

  // 删除房间 - 移除user字段，添加更多调试信息和错误处理
  const deleteRoom = async (roomIdToDelete: number) => {
    if (isDeletingRoom) {
      console.log("正在删除房间，跳过重复请求");
      return;
    }

    if (roomListData?.rooms && roomListData.rooms.length <= 1) {
      alert("至少需要保留一个房间！");
      closeContextMenu();
      return;
    }

    const currentRooms = roomListData?.rooms || [];
    const roomToDelete = currentRooms.find((r) => r.roomId === roomIdToDelete);

    if (!roomToDelete) {
      console.log("房间不在当前列表中，roomIdToDelete:", roomIdToDelete);
      console.log(
        "当前房间列表:",
        currentRooms.map((r) => ({ id: r.roomId, name: r.roomName }))
      );
      alert("房间不存在或已被删除！");
      closeContextMenu();
      await mutateRooms();
      return;
    }

    if (!confirm(`确定要删除房间"${roomToDelete.roomName}"吗？`)) {
      closeContextMenu();
      return;
    }

    closeContextMenu();

    try {
      console.log("开始删除房间:", {
        roomIdToDelete,
        roomName: roomToDelete.roomName,
        currentRoomsCount: currentRooms.length,
      });

      // 如果要删除的是当前房间，先切换到其他房间
      if (roomId === roomIdToDelete) {
        const remainingRooms = currentRooms.filter(
          (r) => r.roomId !== roomIdToDelete
        );
        if (remainingRooms.length > 0) {
          console.log("切换到房间:", remainingRooms[0].roomId);
          setRoomId(remainingRooms[0].roomId);
          await new Promise((resolve) => setTimeout(resolve, 200)); // 增加等待时间
        } else {
          setRoomId(null);
        }
      }

      // 调用删除API - 移除user字段
      const deleteParams = {
        roomId: roomIdToDelete,
      };

      console.log("调用删除API，参数:", deleteParams);

      await deleteRoomTrigger(deleteParams);

      console.log("房间删除API调用成功:", roomIdToDelete);

      // 删除成功后刷新房间列表
      await mutateRooms();

      console.log("房间列表已刷新");
    } catch (error) {
      console.error("删除房间失败 - 详细错误信息:", {
        error,
        errorMessage: error instanceof Error ? error.message : String(error),
        errorName: error instanceof Error ? error.name : "Unknown",
        errorStack: error instanceof Error ? error.stack : "No stack",
        roomId: roomIdToDelete,
      });

      const errorMessage =
        error instanceof Error ? error.message : String(error);

      // 更精确的错误处理
      if (
        errorMessage.includes("房间不存在") ||
        errorMessage.includes("不存在") ||
        errorMessage.includes("404") ||
        errorMessage.includes("请求失败")
      ) {
        console.log("房间可能已经被删除，刷新房间列表");
        alert("房间可能已经被删除或不存在！正在刷新列表...");
        await mutateRooms();
      } else if (errorMessage.includes("至少需要保留一个房间")) {
        alert("至少需要保留一个房间！");
      } else {
        alert(`删除房间失败: ${errorMessage}`);
      }
    }
  };

  // 获取当前房间数据 - 确保始终显示聊天界面而不是空状态
  const getCurrentRoomData = (): ChatRoomData | null => {
    // 只有在没有房间或房间列表为空时才返回null
    if (!roomListData?.rooms || roomListData.rooms.length === 0) {
      return null;
    }

    // 如果还没选择房间，返回null（但这种情况应该很快就会被useEffect处理）
    if (roomId === null) {
      return null;
    }

    const currentRoom = roomListData.rooms.find((r) => r.roomId === roomId);
    if (!currentRoom) {
      // 如果当前选择的房间不存在了，自动选择第一个房间
      console.log("当前房间不存在，自动切换到第一个房间");
      setRoomId(roomListData.rooms[0].roomId);
      return null;
    }

    // 即使消息为空也显示聊天界面
    const messages = messageListData?.messages || [];

    return {
      roomId,
      roomName: currentRoom.roomName,
      messages: messages,
      participants: [],
    };
  };

  const currentRoomData = getCurrentRoomData();

  // 处理全局点击，关闭右键菜单
  useEffect(() => {
    const handleGlobalClick = () => {
      if (contextMenu.visible) {
        closeContextMenu();
      }
    };

    document.addEventListener("click", handleGlobalClick);
    return () => {
      document.removeEventListener("click", handleGlobalClick);
    };
  }, [contextMenu.visible]);

  // 加载状态
  if (roomIsLoading) {
    return (
      <main className="chatroom-main">
        <div className="loading-state">
          <div className="loading-content">
            <div className="loading-spinner">⏳</div>
            <p>正在加载聊天室...</p>
          </div>
        </div>
      </main>
    );
  }

  // 错误状态
  if (roomError) {
    return (
      <main className="chatroom-main">
        <div className="error-state">
          <div className="error-content">
            <div className="error-icon">❌</div>
            <h3>加载失败</h3>
            <p>无法连接到服务器: {roomError.message}</p>
            <button onClick={() => mutateRooms()} className="retry-button">
              重试
            </button>
          </div>
        </div>
      </main>
    );
  }

  return (
    <main className="chatroom-main">
      {/* 左侧边栏 - 房间列表 */}
      <aside className="room-sidebar">
        <header className="sidebar-header">
          <h1 className="sidebar-title">聊天房间</h1>
          <div className="user-status">
            <span className="current-user-indicator">{currentUser}</span>
            <div className="online-status"></div>
          </div>
        </header>

        <nav className="room-navigation">
          <div className="create-room-section">
            {!showCreateForm ? (
              <button
                className="create-room-button"
                onClick={() => setShowCreateForm(true)}
                disabled={isCreatingRoom}
              >
                ➕ 创建新房间
              </button>
            ) : (
              <div className="create-room-form">
                <input
                  type="text"
                  className="room-name-input"
                  value={newRoomName}
                  onChange={(e) => setNewRoomName(e.target.value)}
                  placeholder="输入房间名称..."
                  disabled={isCreatingRoom}
                  onKeyDown={(e) => {
                    if (e.key === "Enter" && !isCreatingRoom) {
                      createNewRoom();
                    } else if (e.key === "Escape") {
                      setShowCreateForm(false);
                      setNewRoomName("");
                    }
                  }}
                />
                <div className="form-buttons">
                  <button
                    className="confirm-button"
                    onClick={createNewRoom}
                    disabled={!newRoomName.trim() || isCreatingRoom}
                  >
                    {isCreatingRoom ? "⏳" : "✓"}
                  </button>
                  <button
                    className="cancel-button"
                    onClick={() => {
                      setShowCreateForm(false);
                      setNewRoomName("");
                    }}
                    disabled={isCreatingRoom}
                  >
                    ✗
                  </button>
                </div>
              </div>
            )}
          </div>

          <ul className="room-list">
            {roomListData?.rooms?.map((room) => (
              <li key={room.roomId} className="room-list-item">
                <RoomEntry
                  room={room}
                  isActive={room.roomId === roomId}
                  onClick={handleRoomClick}
                  onContextMenu={handleRoomContextMenu}
                />
              </li>
            ))}
          </ul>
        </nav>
      </aside>

      {/* 右侧主体 - 聊天区域 */}
      <section className="chat-section">
        {currentRoomData ? (
          <>
            <header className="chat-header">
              <div className="room-info">
                <h2 className="room-title">{currentRoomData.roomName}</h2>
                <p className="participant-count">
                  {currentRoomData.messages.length} 条消息
                  {messageListIsLoading && " (更新中...)"}
                </p>
              </div>
              <div className="room-actions">
                <button
                  className="action-button"
                  type="button"
                  aria-label="房间设置"
                >
                  ⚙️
                </button>
              </div>
            </header>

            <div className="messages-viewport">
              <div className="messages-container">
                {messageListError ? (
                  <div className="error-message">
                    <p>消息加载失败: {messageListError.message}</p>
                    <button onClick={() => mutateMessages()}>重试</button>
                  </div>
                ) : messageListIsLoading &&
                  currentRoomData.messages.length === 0 ? (
                  <div className="loading-messages">
                    <p>正在加载消息...</p>
                  </div>
                ) : currentRoomData.messages.length === 0 ? (
                  <div className="no-messages">
                    <p>还没有消息，发送第一条消息开始聊天吧！</p>
                  </div>
                ) : (
                  currentRoomData.messages.map((msg) => (
                    <MessageItem
                      key={msg.messageId}
                      message={msg}
                      isCurrentUser={msg.sender === currentUser}
                    />
                  ))
                )}
              </div>
            </div>

            <footer className="message-input-section">
              <form
                className="message-form"
                onSubmit={(e) => {
                  e.preventDefault();
                  sendMessage();
                }}
              >
                <div className="input-wrapper">
                  <textarea
                    className="message-textarea"
                    value={message}
                    onChange={(e) => setMessage(e.target.value)}
                    placeholder="输入消息... (按 Enter 发送，Shift+Enter 换行)"
                    rows={1}
                    disabled={isSendingMessage}
                    onKeyDown={(e) => {
                      if (e.key === "Enter" && !e.shiftKey) {
                        e.preventDefault();
                        sendMessage();
                      }
                    }}
                  />
                  <button
                    className="send-button"
                    type="submit"
                    disabled={!message.trim() || isSendingMessage}
                    aria-label="发送消息"
                  >
                    {isSendingMessage ? "⏳" : "📤"}
                  </button>
                </div>
              </form>
            </footer>
          </>
        ) : (
          <div className="empty-state">
            <div className="empty-content">
              <div className="empty-icon">💬</div>
              <h3 className="empty-title">
                {roomListData?.rooms && roomListData.rooms.length === 0
                  ? "还没有聊天房间"
                  : "正在加载聊天室..."}
              </h3>
              <p className="empty-description">
                {roomListData?.rooms && roomListData.rooms.length === 0
                  ? '点击 "创建新房间" 开始您的第一次聊天吧！'
                  : "请稍候，正在为您准备聊天环境..."}
              </p>
            </div>
          </div>
        )}
      </section>

      {/* 右键菜单 */}
      {contextMenu.visible && contextMenu.roomId && (
        <>
          <div className="context-menu-overlay" onClick={closeContextMenu} />
          <div
            className="context-menu"
            style={{
              left: contextMenu.x,
              top: contextMenu.y,
            }}
            onClick={(e) => e.stopPropagation()}
          >
            <button
              className="context-menu-item delete"
              onClick={() => {
                if (contextMenu.roomId && !isDeletingRoom) {
                  deleteRoom(contextMenu.roomId);
                }
              }}
              disabled={isDeletingRoom}
            >
              {isDeletingRoom ? "⏳ 删除中..." : "🗑️ 删除房间"}
            </button>
          </div>
        </>
      )}
    </main>
  );
}
