// ProtectedRoute 是一个 React 组件，作用是保护某些页面，只有登录用户才能访问。
"use client";

import React, { useState, useEffect } from "react";
import { useAuth } from "@/hooks/useAuth";
// 引入 AuthModal 组件，用于显示登录/注册弹窗
// AuthModal 是一个现成的弹窗工具，当用户没登录时，会弹出让用户登录或注册
import AuthModal from "./AuthModal";

// 用 TypeScript 定义了一个 ProtectedRouteProps 接口，规定了 ProtectedRoute 组件可以接收的“参数”（props）。
interface ProtectedRouteProps {
  children: React.ReactNode;
  fallback?: React.ReactNode;
}

export default function ProtectedRoute({
  children,
  fallback,
}: ProtectedRouteProps) {
  // 从 useAuth 钩子中获取 isAuthenticated，判断用户是否已登录
  const { isAuthenticated } = useAuth();
  const [showAuthModal, setShowAuthModal] = useState(false);
  const [isClient, setIsClient] = useState(false);

  useEffect(() => {
    setIsClient(true);
  }, []);

  useEffect(() => {
    if (isClient && !isAuthenticated) {
      setShowAuthModal(true);
    } else {
      setShowAuthModal(false);
    }
  }, [isAuthenticated, isClient]);

  // 服务端渲染时不显示任何内容
  if (!isClient) {
    return null;
  }

  if (!isAuthenticated) {
    return (
      <>
        {fallback || (
          <div style={{ textAlign: "center", padding: "2rem" }}>
            <h2>请先登录</h2>
            <p>您需要登录才能访问聊天室</p>
          </div>
        )}
        <AuthModal
          isOpen={showAuthModal}
          onClose={() => setShowAuthModal(false)}
        />
      </>
    );
  }

  return <>{children}</>;
}
