import { NextResponse } from "next/server";
import { extractUserFromRequest } from "@/lib/auth";
import { RoomAddArgs, RoomAddRes } from "@/types/api";
import { Message } from "@/types";
import {
  getNextRoomId,
  getNextMessageId,
  addRoom,
  addMessage,
  rooms,
} from "@/lib/dataStore";

export async function POST(request: Request) {
  try {
    // 验证用户身份
    const user = extractUserFromRequest(request);
    if (!user) {
      return NextResponse.json({
        code: 1,
        message: "请先登录",
      });
    }

    const body = await request.json();
    const { roomName } = body;

    console.log("创建房间请求:", {
      user: user.username,
      roomName,
      currentRoomsCount: rooms.length,
    });

    if (!roomName) {
      console.log("创建房间参数验证失败: roomName 为空");
      return NextResponse.json({
        code: 1,
        message: "房间名称不能为空",
      });
    }

    const newRoomId = getNextRoomId();
    console.log("生成新房间ID:", newRoomId);

    // 创建欢迎消息
    const welcomeMessage: Message = {
      messageId: getNextMessageId(),
      roomId: newRoomId,
      sender: "系统",
      content: `欢迎来到${roomName}房间！`,
      time: Date.now(),
    };

    console.log("创建欢迎消息:", welcomeMessage);

    // 创建新房间 - 使用你原有的接口格式
    const newRoom = {
      roomId: newRoomId,
      roomName: roomName.trim(),
      lastMessage: welcomeMessage,
    };

    console.log("准备添加房间:", newRoom);

    // 添加房间和欢迎消息
    addRoom(newRoom);
    addMessage(welcomeMessage);

    console.log("房间和消息添加完成");

    const response: RoomAddRes = { roomId: newRoomId };

    console.log("创建房间成功:", response);

    return NextResponse.json({
      code: 0,
      data: response,
    });
  } catch (error) {
    console.error("创建房间API异常:", error);
    return NextResponse.json({
      code: 1,
      message:
        "创建房间失败: " +
        (error instanceof Error ? error.message : String(error)),
    });
  }
}
