import { NextRequest, NextResponse } from 'next/server';
import { userStore } from '@/lib/dataStore';
import { hashPassword, generateToken } from '@/lib/auth';
import { RegisterArgs, AuthResponse } from '@/types/auth';
import { ApiResponse } from '@/types/api';

export async function POST(request: NextRequest) {
  try {
    const body: RegisterArgs = await request.json();
    const { username, password, email } = body;

    // 验证输入
    if (!username || !password) {
      return NextResponse.json<ApiResponse>({
        code: 1,
        message: '用户名和密码不能为空',
      });
    }

    if (username.length < 3 || password.length < 6) {
      return NextResponse.json<ApiResponse>({
        code: 1,
        message: '用户名至少3位，密码至少6位',
      });
    }

    // 检查用户名是否已存在
    if (userStore.isUsernameExists(username)) {
      return NextResponse.json<ApiResponse>({
        code: 1,
        message: '用户名已存在',
      });
    }

    // 创建用户
    const passwordHash = await hashPassword(password);
    const user = await userStore.createUser(username, email, passwordHash);

    // 生成 token
    const token = generateToken({
      userId: user.userId,
      username: user.username,
    });

    const response: AuthResponse = {
      user: {
        userId: user.userId,
        username: user.username,
        email: user.email,
      },
      token,
    };

    return NextResponse.json<ApiResponse<AuthResponse>>({
      code: 0,
      data: response,
      message: '注册成功',
    });

  } catch (error) {
    console.error('注册失败:', error);
    return NextResponse.json<ApiResponse>({
      code: 1,
      message: '服务器错误',
    });
  }
}
