import { NextRequest, NextResponse } from 'next/server';
import { userStore } from '@/lib/dataStore';
import { verifyPassword, generateToken } from '@/lib/auth';
import { LoginArgs, AuthResponse } from '@/types/auth';
import { ApiResponse } from '@/types/api';

export async function POST(request: NextRequest) {
  try {
    const body: LoginArgs = await request.json();
    const { username, password } = body;

    // 验证输入
    if (!username || !password) {
      return NextResponse.json<ApiResponse>({
        code: 1,
        message: '用户名和密码不能为空',
      });
    }

    // 查找用户
    const user = userStore.findUserByUsername(username);
    if (!user) {
      return NextResponse.json<ApiResponse>({
        code: 1,
        message: '用户名或密码错误',
      });
    }

    // 验证密码
    const isPasswordValid = await verifyPassword(password, user.passwordHash);
    if (!isPasswordValid) {
      return NextResponse.json<ApiResponse>({
        code: 1,
        message: '用户名或密码错误',
      });
    }

    // 生成 token
    const token = generateToken({
      userId: user.userId,
      username: user.username,
    });

    const response: AuthResponse = {
      user: {
        userId: user.userId,
        username: user.username,
        email: user.email,
      },
      token,
    };

    return NextResponse.json<ApiResponse<AuthResponse>>({
      code: 0,
      data: response,
      message: '登录成功',
    });

  } catch (error) {
    console.error('登录失败:', error);
    return NextResponse.json<ApiResponse>({
      code: 1,
      message: '服务器错误',
    });
  }
}
