// 用户认证系统，使用 JWT 进行身份验证，并使用 bcrypt 加密和验证密码。
// jsonwebtoken 是一个用于处理 JSON Web Token（JWT）的库
import jwt from "jsonwebtoken";
// bcryptjs 是一个用于加密和解密密码的库
import bcrypt from "bcryptjs";
// 这里导入了我们自己定义的类型 JWTPayload 和 AuthenticatedUser，用于描述用户的 JWT 载荷（payload）和已认证的用户信息。
import { JWTPayload, AuthenticatedUser } from "@/types/auth";

const JWT_SECRET =
  process.env.JWT_SECRET || "your-secret-key-change-in-production";
const JWT_EXPIRES_IN = "7d";

// 生成 JWT token
export function generateToken(user: AuthenticatedUser): string {
  return jwt.sign(
    {
      userId: user.userId,
      username: user.username,
    },
    JWT_SECRET,
    { expiresIn: JWT_EXPIRES_IN }
  );
}

// 验证 JWT token
export function verifyToken(token: string): JWTPayload | null {
  try {
    return jwt.verify(token, JWT_SECRET) as JWTPayload;
  } catch (error) {
    return null;
  }
}

// 密码加密
// 这是一个异步函数，用于加密用户的密码。它接受一个明文的 password 作为参数，并返回加密后的密码。
export async function hashPassword(password: string): Promise<string> {
  return bcrypt.hash(password, 12);
}

// 密码验证
// 这个函数用于验证用户输入的密码是否与存储在数据库中的加密密码（hashedPassword）相匹配。它返回一个布尔值（true 或 false）。
export async function verifyPassword(
  password: string,
  hashedPassword: string
): Promise<boolean> {
  return bcrypt.compare(password, hashedPassword);
}

// 从请求头中提取用户信息
export function extractUserFromRequest(
  request: Request
): AuthenticatedUser | null {
  const authHeader = request.headers.get("Authorization");
  if (!authHeader || !authHeader.startsWith("Bearer ")) {
    return null;
  }

  const token = authHeader.substring(7);
  const payload = verifyToken(token);

  if (!payload) {
    return null;
  }

  return {
    userId: payload.userId,
    username: payload.username,
  };
}
