"use client";

import React, { useState } from "react";
import { useAuth } from "@/hooks/useAuth";
import "./Auth.css";

interface RegisterFormProps {
  onSuccess?: () => void;
  switchToLogin: () => void;
}

export default function RegisterForm({
  onSuccess,
  switchToLogin,
}: RegisterFormProps) {
  // 定义状态变量：用户名、邮箱、密码、确认密码、加载状态和错误信息
  const [username, setUsername] = useState("");
  const [email, setEmail] = useState("");
  const [password, setPassword] = useState("");
  const [confirmPassword, setConfirmPassword] = useState("");
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState("");

  // 从 useAuth 钩子中获取 login 函数，用于处理注册后的登录
  const { login } = useAuth();

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!username.trim() || !password.trim()) {
      setError("请输入用户名和密码");
      return;
    }

    // 检查两次输入的密码是否一致
    if (password !== confirmPassword) {
      setError("两次输入的密码不一致");
      return;
    }

    if (username.length < 3) {
      setError("用户名至少需要3个字符");
      return;
    }

    if (password.length < 6) {
      setError("密码至少需要6个字符");
      return;
    }

    setLoading(true);
    setError("");

    try {
      // 使用 fetch 函数发送 HTTP POST 请求到 /api/auth/register 接口，携带用户名、密码和邮箱（可选）。
      const response = await fetch("/api/auth/register", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          username,
          password,
          email: email.trim() || undefined,
        }),
      });

      const data = await response.json();

      //检查服务器返回的数据：如果 data.code 是 0 且 data.data 存在，说明注册成功。
      if (data.code === 0 && data.data) {
        login(data.data);
        onSuccess?.();
      } else {
        setError(data.message || "注册失败");
      }
    } catch (error) {
      setError("网络错误，请重试");
      console.error("Register error:", error);
    } finally {
      setLoading(false);
    }
  };

  return (
    <form onSubmit={handleSubmit} className="auth-form">
      <h2>用户注册</h2>

      {error && <div className="auth-error">{error}</div>}

      <div className="auth-field">
        <label htmlFor="username">用户名 *</label>
        <input
          id="username"
          type="text"
          value={username}
          onChange={(e) => setUsername(e.target.value)}
          placeholder="请输入用户名（至少3位）"
          disabled={loading}
        />
      </div>

      <div className="auth-field">
        <label htmlFor="email">邮箱 (可选)</label>
        <input
          id="email"
          type="email"
          value={email}
          onChange={(e) => setEmail(e.target.value)}
          placeholder="请输入邮箱地址"
          disabled={loading}
        />
      </div>

      <div className="auth-field">
        <label htmlFor="password">密码 *</label>
        <input
          id="password"
          type="password"
          value={password}
          onChange={(e) => setPassword(e.target.value)}
          placeholder="请输入密码（至少6位）"
          disabled={loading}
        />
      </div>

      <div className="auth-field">
        <label htmlFor="confirmPassword">确认密码 *</label>
        <input
          id="confirmPassword"
          type="password"
          value={confirmPassword}
          onChange={(e) => setConfirmPassword(e.target.value)}
          placeholder="请再次输入密码"
          disabled={loading}
        />
      </div>

      <button type="submit" className="auth-button" disabled={loading}>
        {loading ? "注册中..." : "注册"}
      </button>

      <div className="auth-switch">
        已有账户？
        <button type="button" onClick={switchToLogin} className="auth-link">
          立即登录
        </button>
      </div>
    </form>
  );
}
