import { NextResponse } from "next/server";
import { extractUserFromRequest } from "@/lib/auth";
import { RoomDeleteArgs } from "@/types/api";
import { rooms, removeRoom } from "@/lib/dataStore";

export async function POST(request: Request) {
  try {
    // 验证用户身份
    const user = extractUserFromRequest(request);
    if (!user) {
      return NextResponse.json({
        code: 1,
        message: "请先登录",
      });
    }

    const body = await request.json();
    const { roomId } = body;

    console.log("删除房间请求:", {
      user: user.username,
      roomId,
      currentRoomsCount: rooms.length,
    });

    if (roomId === undefined || roomId === null) {
      console.log("参数验证失败: roomId 为空");
      return NextResponse.json({
        code: 1,
        message: "房间ID不能为空",
      });
    }

    // 检查是否至少保留一个房间（提前检查）
    if (rooms.length <= 1) {
      console.log("房间数量不足，无法删除");
      return NextResponse.json({
        code: 1,
        message: "至少需要保留一个房间",
      });
    }

    // 确保 roomId 是数字类型
    const roomIdToDelete = Number(roomId);
    if (isNaN(roomIdToDelete)) {
      console.log("房间ID格式错误:", roomId);
      return NextResponse.json({
        code: 1,
        message: "房间ID格式错误",
      });
    }

    // 检查房间是否存在
    const roomExists = rooms.some((room) => room.roomId === roomIdToDelete);
    console.log("房间存在性检查:", {
      roomIdToDelete,
      exists: roomExists,
    });

    if (!roomExists) {
      console.log(`房间 ${roomIdToDelete} 不存在，视为已删除`);
      return NextResponse.json({
        code: 0,
        message: "房间已删除或不存在",
      });
    }

    console.log(`开始删除房间 ${roomIdToDelete}`);
    const success = removeRoom(roomIdToDelete);

    console.log("删除结果:", {
      success,
      remainingRoomsCount: rooms.length,
    });

    if (!success) {
      console.log("删除房间操作失败");
      return NextResponse.json({
        code: 1,
        message: "删除房间失败",
      });
    }

    console.log(`房间 ${roomIdToDelete} 删除成功`);
    return NextResponse.json({
      code: 0,
      message: "删除成功",
    });
  } catch (error) {
    console.error("删除房间API异常:", error);
    return NextResponse.json({
      code: 1,
      message:
        "删除房间失败: " +
        (error instanceof Error ? error.message : String(error)),
    });
  }
}
