// chatroom/src/app/chat/page.tsx
'use client';

import { useEffect, useState } from 'react';
import { useRouter } from 'next/navigation';
import useSWR from 'swr';
import useSWRMutation from 'swr/mutation';
import styles from './page.module.css';
import { authFetch } from '@/lib/fetcher';
import { useAuth } from '@/hooks/useAuth';

// 类型定义
interface Message {
  messageId: number;
  roomId: number;
  sender: string;
  content: string;
  time: number;
}
interface RoomPreviewInfo {
  roomId: number;
  roomName: string;
  lastMessage: Message | null;
}
interface RoomListRes {
  rooms: RoomPreviewInfo[];
}
interface RoomAddArgs {
  user: string;
  roomName: string;
}
interface RoomAddRes {
  roomId: number;
}
interface RoomMessageListRes {
  messages: Message[];
}
interface MessageAddArgs {
  roomId: number;
  content: string;
  sender: string;
}

// 单条消息组件
interface MessageItemProps {
  message: Message;
  currentUser: string;
}

function MessageItem({ message, currentUser }: MessageItemProps) {
  const isOwn = message.sender === currentUser;
  
  // 格式化时间为 "年-月-日 时:分" 格式
  const formatTime = (timestamp: number) => {
    const date = new Date(timestamp);
    const year = date.getFullYear();
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const day = String(date.getDate()).padStart(2, '0');
    const hours = String(date.getHours()).padStart(2, '0');
    const minutes = String(date.getMinutes()).padStart(2, '0');
    
    return `${year}-${month}-${day} ${hours}:${minutes}`;
  };
  
  return (
    <div className={`${styles.messageItem} ${isOwn ? styles.ownMessage : styles.otherMessage}`}>
      <div className={styles.messageHeader}>
        <span className={styles.sender}>{message.sender}</span>
        <span className={styles.time}>{formatTime(message.time)}</span>
      </div>
      <div className={styles.messageContent}>{message.content}</div>
    </div>
  );
}

// 单个房间组件
// 登录状态组件
function AuthStatus({ isLoggedIn, username, onLogout }: { 
  isLoggedIn: boolean; 
  username: string | null; 
  onLogout: () => void;
}) {
  return (
    <div className={styles.authStatus}>
      {isLoggedIn ? (
        <div className={styles.loggedIn}>
          <span className={styles.userInfo}>已登录: {username}</span>
          <button 
            className={styles.logoutBtn}
            onClick={onLogout}
            title="退出登录"
          >
            退出
          </button>
        </div>
      ) : (
        <div className={styles.notLoggedIn}>
          <span className={styles.guestInfo}>游客模式</span>
          <a href="/login" className={styles.loginLink}>
            去登录
          </a>
        </div>
      )}
    </div>
  );
}

// 权限提示组件
function PermissionAlert({ message, onClose }: { message: string; onClose: () => void }) {
  return (
    <div className={styles.permissionAlert}>
      <div className={styles.alertContent}>
        <p>{message}</p>
        <div className={styles.alertButtons}>
          <button onClick={onClose} className={styles.alertCloseBtn}>
            知道了
          </button>
          <a href="/login" className={styles.alertLoginBtn}>
            去登录
          </a>
        </div>
      </div>
    </div>
  );
}

// 单个房间组件 - 添加权限检查
interface RoomEntryProps {
  room: RoomPreviewInfo;
  onClick: () => void;
  isActive: boolean;
  onDelete: () => void;
  canDelete: boolean; // 新增：是否可以删除
}

function RoomEntry({ room, onClick, isActive, onDelete, canDelete }: RoomEntryProps) {
  const handleDeleteClick = (e: React.MouseEvent) => {
    e.stopPropagation();
    onDelete();
  };

  return (
    <div
      className={`${styles.roomItem} ${isActive ? styles.active : ''}`}
      onClick={onClick}
    >
      <h3 className={styles.roomName}>{room.roomName}</h3>
      <p className={styles.lastMessage}>
        {room.lastMessage
          ? `${room.lastMessage.sender}: ${room.lastMessage.content}`
          : '暂无消息'}
      </p>
      {canDelete && (
        <button
          className={styles.deleteBtn}
          onClick={handleDeleteClick}
          title="删除房间"
        >
          ×
        </button>
      )}
      {!canDelete && (
        <div className={styles.noPermissionHint} title="需要登录才能删除房间">
          {/* 可以添加一个禁用状态的图标 */}
        </div>
      )}
    </div>
  );
}

export default function ChatPage() {
  const [nickname, setNickname] = useState('');
  const [selectedRoomId, setSelectedRoomId] = useState<number | null>(null);
  const [showAddModal, setShowAddModal] = useState(false);
  const [newRoomName, setNewRoomName] = useState('');
  const [messageInput, setMessageInput] = useState('');
  const [showPermissionAlert, setShowPermissionAlert] = useState(false);
  const [permissionAlertMessage, setPermissionAlertMessage] = useState('');
  const router = useRouter();

  // 使用身份验证 hook
  const { isLoggedIn, username: authUsername, isLoading: authLoading, logout } = useAuth();

  useEffect(() => {
    const saved = localStorage.getItem('nickname');
    if (!saved) {
      alert('请先设置昵称');
      router.push('/setname');
    } else {
      setNickname(saved);
    }
  }, [router]);

  // 检查权限的辅助函数
  const checkPermissionAndAlert = (action: string): boolean => {
    if (!isLoggedIn) {
      setPermissionAlertMessage(`${action}功能需要登录后才能使用，当前为游客模式。`);
      setShowPermissionAlert(true);
      return false;
    }
    return true;
  };

  // 获取房间列表
  const {
    data: roomData,
    error: roomError,
    isLoading: roomIsLoading,
    mutate: refreshRooms,
  } = useSWR<RoomListRes>(
    '/api/room/list',
    (key) => authFetch(key).then((r) => r.json().then((d: any) => d.data)),
    { refreshInterval: 1000 }
  );

  // 获取消息列表
  const {
    data: messageData,
    error: messageError,
    isLoading: messageIsLoading,
    mutate: refreshMessages,
  } = useSWR<RoomMessageListRes>(
    selectedRoomId ? `/api/room/message/list?roomId=${selectedRoomId}` : null,
    (key) => authFetch(key).then((r) => r.json().then((d: any) => d.data)),
    { refreshInterval: 1000 }
  );

  // 创建房间
  const { trigger: addRoomTrigger, isMutating: isAddingRoom } = useSWRMutation<
    RoomAddRes,
    Error,
    string,
    RoomAddArgs
  >('/api/room/add', (key, { arg }) =>
    authFetch(key, {
      method: 'POST',
      body: JSON.stringify(arg),
    }).then((r) => r.json().then((d: any) => d.data))
  );

  // 删除房间
  const { trigger: deleteRoomTrigger } = useSWRMutation(
  '/api/room/delete', // 确保路径正确
  async (key, { arg }: { arg: { roomId: number } }) => {
    const response = await authFetch(key, {
      method: 'POST',
      body: JSON.stringify({ roomId: arg.roomId }), // 参数格式修正
    });
    return response;
  }
);

  // 发送消息
  const { trigger: sendMessageTrigger, isMutating: isSendingMessage } =
    useSWRMutation<null, Error, string, MessageAddArgs>(
      '/api/message/add',
      (key, { arg }) =>
        authFetch(key, {
          method: 'POST',
          body: JSON.stringify(arg),
        }).then((r) => r.json().then((d: any) => d.data))
    );

  // 处理显示添加房间模态框
  const handleShowAddModal = () => {
    if (!checkPermissionAndAlert('创建房间')) return;
    setShowAddModal(true);
  };

  // 处理创建房间
  const handleAddRoom = async () => {
    if (!newRoomName.trim()) return alert('请输入房间名称');
    
    if (!checkPermissionAndAlert('创建房间')) {
      setShowAddModal(false);
      return;
    }

    try {
      const res = await addRoomTrigger({ user: nickname, roomName: newRoomName.trim() });
      refreshRooms();
      if (res?.roomId) setSelectedRoomId(res.roomId);
      setShowAddModal(false);
      setNewRoomName('');
    } catch (error) {
      console.error('创建房间失败:', error);
      alert('创建房间失败');
    }
  };

  // 处理删除房间
  const handleDeleteRoom = async (roomId: number) => {
  if (!isLoggedIn) {
    alert('请先登录后再删除房间');
    return;
  }

  const token = localStorage.getItem('token');
  if (!token) {
    alert('登录状态已过期，请重新登录');
    logout();
    return;
  }

  if (!confirm('确定删除该房间？此操作不可撤销！')) return;
  
  try {
    await deleteRoomTrigger({ roomId });
    
    // 删除成功后的处理
    if (selectedRoomId === roomId) {
      setSelectedRoomId(null);
    }
    
    // 立即刷新房间列表
    refreshRooms();
    
    alert('房间删除成功');
  } catch (error) {
    console.error('删除房间失败:', error);
    
    if (error instanceof Error) {
      if (error.message.includes('认证') || error.message.includes('token')) {
        alert('认证失败，请重新登录');
        logout();
      } else {
        alert(`删除房间失败: ${error.message}`);
      }
    } else {
      alert('删除房间失败，请稍后重试');
    }
  }
};


  // 处理发送消息
  const handleSendMessage = async () => {
    if (!messageInput.trim() || !selectedRoomId) return;
    try {
      await sendMessageTrigger({
        roomId: selectedRoomId,
        content: messageInput.trim(),
        sender: nickname,
      });
      setMessageInput('');
      refreshMessages();
    } catch {
      alert('发送消息失败');
    }
  };

  const handleKeyPress = (e: React.KeyboardEvent<HTMLInputElement>) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      handleSendMessage();
    }
  };

  if (authLoading) {
    return <div className={styles.loading}>正在检查登录状态...</div>;
  }

  return (
    <main className={styles.chatContainer}>
      {/* 身份验证状态显示 */}
      <AuthStatus 
        isLoggedIn={isLoggedIn} 
        username={authUsername} 
        onLogout={logout}
      />

      {/* 左侧房间列表 */}
      <div className={styles.sidebar}>
        <div className={styles.sidebarHeader}>
          <h2>聊天室</h2>
          <button 
            className={styles.addButton} 
            onClick={handleShowAddModal}
            title={isLoggedIn ? "创建房间" : "需要登录才能创建房间"}
            disabled={!isLoggedIn}
          >
            +
          </button>
        </div>
        <div className={styles.roomList}>
          {roomIsLoading && <div className={styles.loading}>加载中...</div>}
          {roomError && <div className={styles.error}>加载房间失败</div>}
          {roomData?.rooms?.map((r) => (
            <RoomEntry
              key={r.roomId}
              room={r}
              onClick={() => setSelectedRoomId(r.roomId)}
              isActive={selectedRoomId === r.roomId}
              onDelete={() => handleDeleteRoom(r.roomId)}
              canDelete={isLoggedIn} // 只有登录用户可以删除
            />
          ))}
        </div>
      </div>

      {/* 右侧聊天窗口 */}
      <div className={styles.chatArea}>
        {selectedRoomId ? (
          <div className={styles.chatContent}>
            <div className={styles.chatHeader}>
              <h3>
                {roomData?.rooms.find((r) => r.roomId === selectedRoomId)?.roomName ||
                  `房间 ${selectedRoomId}`}
              </h3>
            </div>
            <div className={styles.messagesContainer}>
              {messageIsLoading && <div className={styles.loading}>加载消息中...</div>}
              {messageError && <div className={styles.error}>加载消息失败</div>}
              {messageData?.messages?.map((m) => (
                <MessageItem key={m.messageId} message={m} currentUser={nickname} />
              ))}
            </div>
            <div className={styles.messageInput}>
              <input
                type="text"
                placeholder="输入消息..."
                value={messageInput}
                onChange={(e) => setMessageInput(e.target.value)}
                onKeyPress={handleKeyPress}
                disabled={isSendingMessage}
                className={styles.textInput}
              />
              <button
                onClick={handleSendMessage}
                disabled={isSendingMessage || !messageInput.trim()}
                className={styles.sendButton}
              >
                {isSendingMessage ? '发送中...' : '发送'}
              </button>
            </div>
          </div>
        ) : (
          <div className={styles.noRoom}>请选择一个聊天室开始聊天</div>
        )}
      </div>

      {/* 创建房间弹窗 */}
      {showAddModal && (
        <div className={styles.modal} onClick={() => setShowAddModal(false)}>
          <div className={styles.modalContent} onClick={(e) => e.stopPropagation()}>
            <h3>创建新房间</h3>
            <input
              type="text"
              placeholder="房间名称"
              value={newRoomName}
              onChange={(e) => setNewRoomName(e.target.value)}
              className={styles.input}
            />
            <div className={styles.modalButtons}>
              <button onClick={handleAddRoom} disabled={isAddingRoom} className={styles.confirmButton}>
                {isAddingRoom ? '创建中...' : '创建'}
              </button>
              <button onClick={() => setShowAddModal(false)} className={styles.cancelButton}>
                取消
              </button>
            </div>
          </div>
        </div>
      )}

      {/* 权限提示弹窗 */}
      {showPermissionAlert && (
        <div className={styles.modal}>
          <div className={styles.modalContent}>
            <PermissionAlert 
              message={permissionAlertMessage}
              onClose={() => setShowPermissionAlert(false)}
            />
          </div>
        </div>
      )}
    </main>
  );
}
