// src/hooks/useAuth.ts
import { useState, useEffect } from 'react';
import jwt from 'jsonwebtoken';

interface DecodedToken {
  sub: string;
  username: string;
  exp: number;
}

export function useAuth() {
  const [isLoggedIn, setIsLoggedIn] = useState(false);
  const [username, setUsername] = useState<string | null>(null);
  const [isLoading, setIsLoading] = useState(true);

  useEffect(() => {
    checkAuthStatus();
  }, []);

  const checkAuthStatus = () => {
    try {
      const token = localStorage.getItem('token');
      
      if (!token) {
        setIsLoggedIn(false);
        setUsername(null);
        setIsLoading(false);
        return;
      }

      // 解码 token 检查是否过期
      const decoded = jwt.decode(token) as DecodedToken;
      
      if (!decoded || !decoded.exp) {
        // 无效的 token
        localStorage.removeItem('token');
        setIsLoggedIn(false);
        setUsername(null);
        setIsLoading(false);
        return;
      }

      // 检查是否过期
      const currentTime = Math.floor(Date.now() / 1000);
      if (decoded.exp < currentTime) {
        // token 过期
        localStorage.removeItem('token');
        setIsLoggedIn(false);
        setUsername(null);
        setIsLoading(false);
        return;
      }

      // token 有效
      setIsLoggedIn(true);
      setUsername(decoded.username);
      setIsLoading(false);
    } catch (error) {
      console.error('检查登录状态失败:', error);
      localStorage.removeItem('token');
      setIsLoggedIn(false);
      setUsername(null);
      setIsLoading(false);
    }
  };

  const logout = () => {
    localStorage.removeItem('token');
    localStorage.removeItem('nickname');
    setIsLoggedIn(false);
    setUsername(null);
  };

  const validateToken = () => {
    try {
      const token = localStorage.getItem('token');
      
      if (!token) {
        return false;
      }

      const decoded = jwt.decode(token) as DecodedToken;
      
      if (!decoded || !decoded.exp) {
        return false;
      }

      const currentTime = Math.floor(Date.now() / 1000);
      return decoded.exp >= currentTime;
    } catch (error) {
      console.error('Token 验证失败:', error);
      return false;
    }
  };

  return {
    isLoggedIn,
    username,
    isLoading,
    checkAuthStatus,
    logout,
    validateToken
  };
}