import { useState, useEffect } from 'react'
import './App.css'

const BackEndUrl : string = 'http://localhost:8080'
const UsePages : boolean = true // 是否使用分页
const PageSize : number = 10 // 每页显示的评论数量

interface CommentData {
  name: string
  content: string
}
interface CommentFromBack {
  id: number
  name: string
  content: string
}

function App() {
  const [username, setUsername] = useState<string>('')
  const [comment, setComment] = useState<string>('')
  const [error, setError] = useState<string>('')
  const [loading, setLoading] = useState<boolean>(false)
  const [comments, setComments] = useState<CommentFromBack[]>([])
  const [page, setPage] = useState<number>(1)
  const [totalPage, setTotalPage]  = useState<number>(1)
  const [commentCount, setCommentCount]  = useState<number>(0)

  const clearForm = () => {
    setUsername('')
    setComment('')
    setError('')
  }

  const handleSubmit = async (e: React.FormEvent) => {
    //无需检查是否为空，有required属性

    //消除默认的表单提交行为
    e.preventDefault()

    // 清除之前的错误信息
    setError('') 

    // 构造评论结构
    const Comment : CommentData = {
      name: username,
      content: comment
    }

    try {
      const response : Response = await fetch(BackEndUrl + '/comment/add', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(Comment),
      })

      const text = await response.text();

      //输出responce 的详细信息到控制台
      console.log('Response Status:', response.status);
      console.log('Response Headers:', response.headers);
      console.log('Response URL:', response.url);
      console.log('Response Body:', text);

      if (!response.ok) {
        throw new Error('无法提交评论，错误码：' + response.status)
      }

      const dataJson = JSON.parse(text)

      if (dataJson.code !== 0) {
        throw new Error('获取评论失败：' + dataJson.msg)
      }

      clearForm()
      //alert('提交成功！');
      fetchComments()

    } catch (err) {
      // 处理错误
      console.error('提交失败:', err)
      setError('提交失败：' + (err instanceof Error ? err.message : '未知错误'))
      alert(error);
      return
    }
  }

  const fetchCommentsPage = async (page : number) => {
    setLoading(true)
    setError('')

    try {

      const params = new URLSearchParams();
      if (UsePages) {
        params.append('page', page.toString());
        params.append('size', PageSize.toString());
      } else {
        params.append('size', '-1'); // -1表示获取所有评论
      }

      const response : Response = await fetch(BackEndUrl + '/comment/get?' + params.toString(), {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
        },
      })

      const text = await response.text();

      //输出responce 的详细信息到控制台
      console.log('Response Status:', response.status);
      console.log('Response Headers:', response.headers);
      console.log('Response URL:', response.url);
      console.log('Response Body:', text);

       if (!response.ok) {
        throw new Error('无法获取评论，错误码：' + response.status)
      }

      const dataJson = JSON.parse(text)

      if (dataJson.code !== 0) {
        throw new Error('获取评论失败：' + dataJson.msg)
      }

      const data : CommentFromBack[] = dataJson.data?.comments || []
      setComments(data)
      console.log('获取到的评论:', data)

      const totalImpl : number= dataJson.data?.total || 1;
      const pageCount : number = Math.max(Math.ceil(totalImpl / PageSize), 1)

      setTotalPage( pageCount  )
      setCommentCount( totalImpl )
      if(page > pageCount)
      {
        setPage ( pageCount )
        fetchCommentsPage ( pageCount )
      }
    } catch (err) {
      // 处理错误
      console.error('获取失败:', err)
      setError('获取失败：' + (err instanceof Error ? err.message : '未知错误'))
      alert(error);
    } finally {
      setLoading(false)
    }

    return
  }

  const fetchComments = async () => fetchCommentsPage(page)

  const deleteComment = async (id: number) => {
    // 确认删除
    if (!window.confirm('确定要删除这条评论吗？')) {
      return
    }

    setError('')

    try {
      const response : Response = await fetch(BackEndUrl + '/comment/delete?id=' + id, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
      })

      //输出responce 的详细信息到控制台
      console.log('Response Status:', response.status);
      console.log('Response Headers:', response.headers);
      console.log('Response URL:', response.url);
      console.log('Response Body:', await response.text());

      // 检查响应状态
       if (!response.ok) {
        throw new Error('无法删除评论，错误码：' + response.status)
      }

    } catch (err) {
      // 处理错误
      console.error('删除失败:', err)
      setError('删除失败：' + (err instanceof Error ? err.message : '未知错误'))
      alert(error);
    } finally {
      // 删除成功后重新获取评论列表
      fetchComments()
    }
  }

  const prevPage = () => {
    if (page > 1) {
      fetchCommentsPage(page - 1)
      setPage(page - 1)
    }
  }

  const nextPage = () => {
    if (page < totalPage){
      fetchCommentsPage(page + 1)
      setPage(page + 1)
    }
  }

  const useFetchComments = () => {
  const [fetchComments] = useState(() => fetchCommentsPage);
  return fetchComments;
};
  const fetchCommentsImpl = useFetchComments();

  useEffect(() => {
    // 初始加载评论
    const loadComments = () => {
      console.log("page = ", page);
      fetchCommentsImpl(page);
    };

    loadComments()

    // 设置定时器，每20秒获取一次评论
    const interval = setInterval(() => {
      loadComments()
    }, 20000) // 20秒

    // 组件卸载时清除定时器
    return () => {
      clearInterval(interval)
    }
  }, [page, fetchCommentsImpl])


  return(
    <>
      <div className="submit-container">
        <form id="todoForm" onSubmit={handleSubmit}>
          <div className='form-text'><h3>用户名称</h3></div>
          <input 
            type="text" 
            id="username" 
            name="username"
            value={username}
            onChange={(e) => setUsername(e.target.value)} 
            required
          ></input>
          <div className='form-text'><h3>评论内容</h3></div>
          <input 
            type="text" 
            id="comment" 
            name="comment"
            value={comment}
            onChange={(e) => setComment(e.target.value)} 
            required
          ></input>
          <div className="submit-button">
            <button type="submit">提交</button>
          </div>
        </form>
      </div>
      <div className="comment-topbar">
        {comments.length !== 0 && <p>共{commentCount}条评论</p>}
        {comments.length === 0 && <p>暂无评论</p>}
        <button type="button" onClick={fetchComments} disabled={loading}>
          {loading ? '↻...' : '↻'}
        </button>
      </div>
      <div className="comments-container">
        {!loading && error && <p className="error">{error}</p>}
        <ul>
          {comments.map((comment) => (
              <li key={comment.id}>
                <div className = "comment-name">
                  <strong>{comment.name}</strong>
                </div>
                <div className = "comment-content">{comment.content}</div>
                <div className = "comment-delete">
                  <button 
                    type="button" 
                    onClick={() => deleteComment(comment.id)}
                  >
                      删除
                    </button>
                </div>
              </li>
            ))}
        </ul>
      </div>
      {
        UsePages && comments.length !== 0 &&
        <div className="page-controls">
        <button onClick={prevPage} disabled={page <= 1}>上一页</button>
        <span>第 {page}/{totalPage} 页</span>
        <button onClick={nextPage} disabled={page >= totalPage}>下一页</button>
        </div>
      }
    </>
  )
}

export default App