import { useState, useEffect } from 'react'
import './App.css'

const BackEndUrl : string = 'http://localhost:8080'

interface CommentData {
  name: string
  content: string
}
interface CommentFromBack {
  id: number
  name: string
  content: string
}

function App() {
  const [username, setUsername] = useState<string>('')
  const [comment, setComment] = useState<string>('')
  const [error, setError] = useState<string>('')
  const [loading, setLoading] = useState<boolean>(false)
  const [comments, setComments] = useState<CommentFromBack[]>([])

  const clearForm = () => {
    setUsername('')
    setComment('')
    setError('')
  }

  const handleSubmit = async (e: React.FormEvent) => {
    //无需检查是否为空，有required属性

    //消除默认的表单提交行为
    e.preventDefault()

    // 清除之前的错误信息
    setError('') 

    // 构造评论结构
    const Comment : CommentData = {
      name: username,
      content: comment
    }

    try {
      const response : Response = await fetch(BackEndUrl + '/comment/add', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(Comment),
      })

      //输出responce 的详细信息到控制台
      console.log('Response Status:', response.status);
      console.log('Response Headers:', response.headers);
      console.log('Response URL:', response.url);
      console.log('Response Body:', await response.text());

      if (!response.ok) {
        throw new Error('无法提交评论，错误码：' + response.status)
      }

      clearForm()
      //alert('提交成功！');
      fetchComments()

    } catch (err) {
      // 处理错误
      console.error('提交失败:', err)
      setError('提交失败：' + (err instanceof Error ? err.message : '未知错误'))
      alert(error);
      return
    }
  }

  const fetchComments = async () => {
    setLoading(true)
    setError('')

    try {
      const response : Response = await fetch('http://localhost:8080/comment/get?size=-1', {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
        },
      })

      const text = await response.text();

      //输出responce 的详细信息到控制台
      console.log('Response Status:', response.status);
      console.log('Response Headers:', response.headers);
      console.log('Response URL:', response.url);
      console.log('Response Body:', text);

       if (!response.ok) {
        throw new Error('无法获取评论，错误码：' + response.status)
      }

      const dataJson = JSON.parse(text)
      /*
         dataJson:
         {"code":0,"msg":"success","data":{"total":4,"comments":[{"id":1,"name":"Alice","content":"First comment!"},{"id":2,"name":"Bob","content":"Great work!"},{"id":3,"name":"114","content":"514"},{"id":4,"name":"114","content":"514"}]}}
      */

      if (dataJson.code !== 0) {
        throw new Error('获取评论失败：' + dataJson.msg)
      }

      const data : CommentFromBack[] = dataJson.data?.comments || []
      setComments(data)
      console.log('获取到的评论:', data)

    } catch (err) {
      // 处理错误
      console.error('提交失败:', err)
      setError('提交失败：' + (err instanceof Error ? err.message : '未知错误'))
      alert(error);
    } finally {
      setLoading(false)
    }

    return
  }

  useEffect(() => {
    // 初始加载评论
    fetchComments()

    // 设置定时器，每20秒获取一次评论
    const interval = setInterval(() => {
      fetchComments()
    }, 20000) // 20秒

    // 组件卸载时清除定时器
    return () => {
      clearInterval(interval)
    }
  }, [])


  return(
    <>
      <div className="submit-container">
        <h3>user name</h3>
        <form id="todoForm" onSubmit={handleSubmit}>
          <input 
            type="text" 
            id="username" 
            name="username"
            value={username}
            onChange={(e) => setUsername(e.target.value)} 
            required
          ></input>
          <h3>comment content</h3>
          <input 
            type="text" 
            id="comment" 
            name="comment"
            value={comment}
            onChange={(e) => setComment(e.target.value)} 
            required
          ></input>
          <div>
            <button type="submit">提交</button>
          </div>
        </form>
      </div>
      <div className="refresh-button">
        <button type="button" onClick={fetchComments} disabled={loading}>
          {loading ? '↻...' : '↻'}
        </button>
      </div>
      <div className="comments-container">
        <h3>评论列表</h3>
        {!loading && error && <p className="error">{error}</p>}
        {!loading && comments.length === 0 && <p>暂无评论</p>}
        <ul>
          {comments.map((comment) => (
              <li key={comment.id}>
                <p>
                  <strong>{comment.name}</strong>
                </p>
                <p>{comment.content}</p>
              </li>
            ))}
        </ul>
      </div>
    </>
  )
}

export default App