package db

import (
	"log"
	"gorm.io/gorm"
)


func GetAllComments() []Comment {
	return GetCommentsDB(DB)
}

func GetCommentsSlice(Begin int, End int) []Comment {
	return GetCommentsSliceDB(DB, Begin, End)
}

func GetCommentCount() int {
	return GetCommentCountDB(DB)
}

func AddComment(Name string, Content string) (Comment, error) {
	return AddCommentDB(DB, Name, Content)
}

func DeleteComment(Id int) bool {
	return DeleteCommentDB(DB, Id)
}


func GetCommentsDB(db *gorm.DB) []Comment {
	var comments []Comment
	if err := db.Find(&comments).Error; err != nil {
		log.Printf("GetComments error: %v", err)
		return nil
	}
	return comments
}

func GetCommentCountDB(db *gorm.DB) int {
	var total int64
	if err := db.Model(&Comment{}).Count(&total).Error; err != nil {
		log.Printf("GetCommentCount error: %v", err)
		return 0
	}
	return int(total)
}

func GetCommentsSliceDB(db *gorm.DB, Begin int, End int) []Comment {
	var comments []Comment
	// 计算总记录数
	var total int = GetCommentCountDB(db)
	if total == 0 {
		return []Comment{} // 如果没有评论，返回空切片
	}
	// 检查边界条件
	if Begin < 0 || total < Begin || total < End || Begin > End {
		return []Comment{}
	}
	// 查询指定范围的评论
	if err := db.Limit(End - Begin).Offset(Begin).Find(&comments).Error; err != nil {
		log.Printf("GetCommentsSlice find error: %v", err)
		return nil
	}
	return comments
}

func AddCommentDB(db *gorm.DB, Name string, Content string) (Comment, error) {
	newComment := Comment{
		Name:    Name,
		Content: Content,
	}
	// 使用事务来确保数据一致性
	tx := db.Begin()
	if err := tx.Create(&newComment).Error; err != nil {
		tx.Rollback()
		log.Printf("AddComment error: %v", err)
		return Comment{}, err
	}
	tx.Commit()
	return newComment, nil
}

func DeleteCommentDB(db *gorm.DB, Id int) bool {
	// 查找并删除评论
	if err := db.Delete(&Comment{}, uint(Id)).Error; err != nil {
		log.Printf("DeleteComment error: %v", err)
		return false
	}
	return true
}