package main

import (
	"encoding/json"
	"log"
	"net/http"
	"strconv"
	"sync"
)

// 统一响应格式
type Response struct {
	Code int         `json:"code"` // 0=成功, 非0=失败
	Msg  string      `json:"msg"`  // 响应消息
	Data interface{} `json:"data"` // 响应数据
}

// 评论数据结构
type Comment struct {
	ID      int    `json:"id"`      // 评论ID
	Name    string `json:"name"`    // 评论者名字
	Content string `json:"content"` // 评论内容
}

// 分页响应结构
type CommentList struct {
	Total    int       `json:"total"`    // 总评论数
	Comments []Comment `json:"comments"` // 评论列表
}

var (
	mutex     sync.RWMutex       // 读写锁保护共享数据
	comments  []Comment          // 内存存储的评论列表
	incrementID int = 1          // 自增ID计数器
)

// 发送统一格式的响应
func sendResponse(w http.ResponseWriter, code int, msg string, data interface{}) {
	w.Header().Set("Content-Type", "application/json")
	resp := Response{
		Code: code,
		Msg:  msg,
		Data: data,
	}
	json.NewEncoder(w).Encode(resp)
}

func respondOptions(w http.ResponseWriter) {
	// 设置CORS头部
	w.Header().Set("Access-Control-Allow-Origin", "*")
	w.Header().Set("Access-Control-Allow-Methods", "GET, POST, OPTIONS")
	w.Header().Set("Access-Control-Allow-Headers", "Content-Type")
	w.WriteHeader(http.StatusOK)
}

// 获取评论处理函数
func getCommentHandler(w http.ResponseWriter, r *http.Request) {

	log.Printf("Received request: %s %s", r.Method, r.URL.Path)
	// 提前处理 OPTIONS 请求
	if r.Method == http.MethodOptions {
		respondOptions(w)
		return
	}

	// 只处理GET请求
	if r.Method != http.MethodGet {
		sendResponse(w, 1, "Method not allowed", nil)
		return
	}

	// 解析查询参数
	page, _ := strconv.Atoi(r.URL.Query().Get("page"))
	size, _ := strconv.Atoi(r.URL.Query().Get("size"))

	// 参数校验
	if page < 1 {
		page = 1
	}
	if size < 1 && size != -1 {
		size = 10
	}

	mutex.RLock()         // 读锁定
	defer mutex.RUnlock() // 函数返回时解锁

	total := len(comments)

	// 返回所有评论
	if size == -1 {
		sendResponse(w, 0, "success", CommentList{
			Total:    total,
			Comments: comments,
		})
		return
	}

	// 计算分页范围
	start := (page - 1) * size
	if start >= total {
		sendResponse(w, 0, "success", CommentList{
			Total:    total,
			Comments: []Comment{},
		})
		return
	}

	end := start + size
	if end > total {
		end = total
	}

	// 返回分页数据
	sendResponse(w, 0, "success", CommentList{
		Total:    total,
		Comments: comments[start:end],
	})
}

// 添加评论处理函数
func addCommentHandler(w http.ResponseWriter, r *http.Request) {
	
	// 输出Request信息到控制台：
	log.Printf("Received request: %s %s", r.Method, r.URL.Path)
	// 提前处理 OPTIONS 请求
	if r.Method == http.MethodOptions {
		respondOptions(w)
		return
	}


	// 只处理POST请求
	if r.Method != http.MethodPost {
		sendResponse(w, 1, "Method not allowed", nil)
		return
	}

	// 解析JSON请求体
	var req struct {
		Name    string `json:"name"`
		Content string `json:"content"`
	}
	
	if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
		sendResponse(w, 2, "Invalid request body", nil)
		return
	}

	log.Printf("Received Info: Name=%s, Content=%s", req.Name, req.Content)

	// 校验必要字段
	if req.Name == "" || req.Content == "" {
		sendResponse(w, 3, "Name and content are required", nil)
		return
	}

	mutex.Lock()         // 写锁定
	defer mutex.Unlock() // 函数返回时解锁

	// 创建新评论
	newComment := Comment{
		ID:      incrementID,
		Name:    req.Name,
		Content: req.Content,
	}
	incrementID++

	// 添加到内存存储
	comments = append(comments, newComment)

	// 返回新创建的评论
	sendResponse(w, 0, "Comment added", newComment)
}

// 删除评论处理函数
func deleteCommentHandler(w http.ResponseWriter, r *http.Request) {

	log.Printf("Received request: %s %s", r.Method, r.URL.Path)
	// 提前处理 OPTIONS 请求
	if r.Method == http.MethodOptions {
		respondOptions(w)
		return
	}

	// 只处理POST请求
	if r.Method != http.MethodPost {
		sendResponse(w, 1, "Method not allowed", nil)
		return
	}

	// 获取评论ID参数
	idStr := r.URL.Query().Get("id")
	if idStr == "" {
		sendResponse(w, 4, "Missing id parameter", nil)
		return
	}

	id, err := strconv.Atoi(idStr)
	if err != nil {
		sendResponse(w, 5, "Invalid id format", nil)
		return
	}

	mutex.Lock()         // 写锁定
	defer mutex.Unlock() // 函数返回时解锁

	// 查找并删除评论
	found := false
	for i, comment := range comments {
		if comment.ID == id {
			// 从切片中删除元素
			comments = append(comments[:i], comments[i+1:]...)
			found = true
			break
		}
	}

	if !found {
		sendResponse(w, 6, "Comment not found", nil)
		return
	}

	sendResponse(w, 0, "Comment deleted", nil)
}

func main() {
	// 初始化一些测试数据
	comments = []Comment{
		{ID: 1, Name: "Alice", Content: "First comment!"},
		{ID: 2, Name: "Bob", Content: "Great work!"},
	}
	incrementID = 3

	// 注册路由
	http.HandleFunc("/comment/get", getCommentHandler)
	http.HandleFunc("/comment/add", addCommentHandler)
	http.HandleFunc("/comment/delete", deleteCommentHandler)

	// 启动服务器
	log.Println("Server starting on http://localhost:8080")
	if err := http.ListenAndServe(":8080", nil); err != nil {
		log.Fatal("Server failed to start: ", err)
	}
}