package main

import (
	"encoding/json"
	"log"
	"net/http"
	"strconv"
	db "BackEnd/db"
)

// 统一响应格式
type Response struct {
	Code int         `json:"code"` // 0=成功, 非0=失败
	Msg  string      `json:"msg"`  // 响应消息
	Data interface{} `json:"data"` // 响应数据
}

// 分页响应结构
type CommentList struct {
	Total    int       `json:"total"`    // 总评论数
	Comments []db.Comment `json:"comments"` // 评论列表
}

// 发送统一格式的响应
func sendResponse(w http.ResponseWriter, code int, msg string, data interface{}) {
	w.Header().Set("Content-Type", "application/json")
	w.Header().Set("Access-Control-Allow-Origin", "*")
    w.Header().Set("Access-Control-Allow-Methods", "GET, POST, OPTIONS")
    w.Header().Set("Access-Control-Allow-Headers", "Content-Type")
	resp := Response{
		Code: code,
		Msg:  msg,
		Data: data,
	}
	json.NewEncoder(w).Encode(resp)
}

func respondOptions(w http.ResponseWriter) {
	// 设置CORS头部
	w.Header().Set("Access-Control-Allow-Origin", "*")
	w.Header().Set("Access-Control-Allow-Methods", "GET, POST, OPTIONS")
	w.Header().Set("Access-Control-Allow-Headers", "Content-Type")
	w.WriteHeader(http.StatusOK)
}

// 获取评论处理函数
func getCommentHandler(w http.ResponseWriter, r *http.Request) {

	log.Printf("Received request: %s %s", r.Method, r.URL.Path)
	// 提前处理 OPTIONS 请求
	if r.Method == http.MethodOptions {
		respondOptions(w)
		return
	}

	// 只处理GET请求
	if r.Method != http.MethodGet {
		sendResponse(w, 1, "Method not allowed", nil)
		return
	}

	// 解析查询参数
	page, _ := strconv.Atoi(r.URL.Query().Get("page"))
	size, _ := strconv.Atoi(r.URL.Query().Get("size"))

	// 参数校验
	if page < 1 {
		page = 1
	}
	if size < 1 && size != -1 {
		size = 10
	}

	total := db.GetCommentCount()

	// 返回所有评论
	if size == -1 {
		sendResponse(w, 0, "success", CommentList{
			Total:    total,
			Comments: db.GetAllComments(),
		})
		return
	}

	// 计算分页范围
	start := (page - 1) * size
	if start >= total {
		sendResponse(w, 0, "success", CommentList{
			Total:    total,
			Comments: []db.Comment{},
		})
		return
	}

	end := start + size
	if end > total {
		end = total
	}

	// 返回分页数据
	sendResponse(w, 0, "success", CommentList{
		Total:    total,
		Comments: db.GetCommentsSlice(start, end),
	})
}

// 添加评论处理函数
func addCommentHandler(w http.ResponseWriter, r *http.Request) {
	
	// 输出Request信息到控制台：
	log.Printf("Received request: %s %s", r.Method, r.URL.Path)
	// 提前处理 OPTIONS 请求
	if r.Method == http.MethodOptions {
		respondOptions(w)
		return
	}


	// 只处理POST请求
	if r.Method != http.MethodPost {
		sendResponse(w, 1, "Method not allowed", nil)
		return
	}

	// 解析JSON请求体
	var req struct {
		Name    string `json:"name"`
		Content string `json:"content"`
	}
	
	if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
		sendResponse(w, 2, "Invalid request body", nil)
		return
	}

	log.Printf("Received Info: Name=%s, Content=%s", req.Name, req.Content)

	// 校验必要字段
	if req.Name == "" || req.Content == "" {
		sendResponse(w, 3, "Name and content are required", nil)
		return
	}

	comment, err := db.AddComment(req.Name, req.Content)
	if err != nil {
		log.Printf("Error adding comment: %v", err)
		sendResponse(w, 4, "Failed to add comment", nil)
		return
	}

	// 返回新创建的评论
	sendResponse(w, 0, "Comment added", comment)
}

// 删除评论处理函数
func deleteCommentHandler(w http.ResponseWriter, r *http.Request) {

	log.Printf("Received request: %s %s", r.Method, r.URL.Path)
	// 提前处理 OPTIONS 请求
	if r.Method == http.MethodOptions {
		respondOptions(w)
		return
	}

	// 只处理POST请求
	if r.Method != http.MethodPost {
		sendResponse(w, 1, "Method not allowed", nil)
		return
	}

	// 获取评论ID参数
	idStr := r.URL.Query().Get("id")
	if idStr == "" {
		sendResponse(w, 4, "Missing id parameter", nil)
		return
	}

	id, err := strconv.Atoi(idStr)
	if err != nil {
		sendResponse(w, 5, "Invalid id format", nil)
		return
	}

	// 查找并删除评论
	found := db.DeleteComment(id)

	if !found {
		sendResponse(w, 6, "Comment not found", nil)
		return
	}

	sendResponse(w, 0, "Comment deleted", nil)
}


func initServer(serverPort string) {
	// 注册路由
	http.HandleFunc("/comment/get", getCommentHandler)
	http.HandleFunc("/comment/add", addCommentHandler)
	http.HandleFunc("/comment/delete", deleteCommentHandler)

	// 启动服务器
	log.Println("Server starting on http://localhost" + serverPort)
	if err := http.ListenAndServe(serverPort, nil); err != nil {
		log.Fatal("Server failed to start: ", err)
	}
}