'use client'

import { useRoom } from './room-context'
import { useEffect, useState } from 'react'
import * as Ty from '../interface'
import * as Auth from '../api/login-interface'
import { maxMessageLength } from '../api/interface-schema'

export default function SendMessageProvider() {

    const { 
        username, 
        currentRoomInfo, 
        setAfterErrorModalEvent,
        triggerErrorModal,
        triggerRefresh 
    } = useRoom()
    const [text, setText] = useState('');
    const [sendError, setSendError] = useState<string | null>(null);
    const [pending, setPending] = useState(false);

    /*
    - url: /api/message/add
    - method: POST
    - argument:
    interface MessageAddArgs {
        roomId: number;
        content: string;
        sender: string;
    }
    - response: null （只要code为0即为成功）
    */
   const sendMessage = async (allowed : boolean) => {
       if (!allowed) return;
       if (pending) return;
       setPending(true);
       
       setSendError(null);

       if (!currentRoomInfo) return;
       if (!text.trim()) return;

       const Args : Ty.MessageAddArgs = {
           roomId: currentRoomInfo.roomId,
           content: text,
           sender: username
       };

       try {
           const response = await fetch(`${Ty.urlPrefix}/api/message/add`, {
               method: 'POST',
               headers: {
                   'Content-Type': 'application/json'
               },
               body: JSON.stringify(Args)
           });

           const data: Ty.BackendResponse<null | Ty.UserNameRedirectArgs> = await response.json();

           if (data.code === 0) {
               setText('');
               triggerRefresh();
           } else {
                if(Ty.useLocalBackend)
                {
                    if(data.code === Auth.incorrectUserName)
                    {
                        const redirectData = data.data as Ty.UserNameRedirectArgs;
                        if (redirectData.user) {
                            const userName = redirectData.user;
                            setAfterErrorModalEvent(() => {
                                window.location.href = `/chat?username=${encodeURIComponent(userName)}`;
                            });
                        }
                    } else if(data.code === Auth.reloginErrorCode)
                    {
                        setAfterErrorModalEvent(() => {
                            window.location.href = `/`;
                        });
                    }
                }
                throw new Error(`发送失败: ${data.message}`);
           }
       } catch (error) {
           setSendError((error as Error).message);
       } finally {
           setPending(false);
       }
   }

    useEffect(() => {
        if (sendError) {
            triggerErrorModal(sendError);
            setSendError(null);
        }
    }, [sendError]);

    const [allowed, setAllowed] = useState(false);
    useEffect(() => {
        setAllowed((currentRoomInfo && text.length && text.length <= maxMessageLength && !pending)?true:false);
    }, [currentRoomInfo, text, pending]);

    return (
        <div className="bg-white p-4">
            <textarea 
                className="w-full resize-none" 
                rows={3} 
                placeholder={currentRoomInfo ? '输入消息...' : '请选择房间后再发送消息'}
                value={text}
                onChange={(e) => setText(e.target.value)}
                disabled={!currentRoomInfo}
            />
            <div className="text-xs text-gray-500 mt-1 ">
                {text.length}/{maxMessageLength}
                {
                    text.length > maxMessageLength && (
                        <span className="text-red-500 ml-2">超出最大长度</span>
                    )
                }
            </div>
            <button 
                className={
                    allowed ?
                    "bg-blue-500 text-white py-2 px-4 mt-2 w-full hover:bg-blue-600 active:bg-blue-700" :
                    "bg-gray-300 text-gray-500 py-2 px-4 mt-2 w-full cursor-not-allowed"
                }
                onClick={() => sendMessage(allowed)}
            >
                发送
            </button>
        </div>
    )
}