'use client';
import { useEffect, useState } from 'react';
import { useRouter } from 'next/navigation';
import { useLocalBackend } from './interface';
import * as Auth from './api/login-interface';
import * as Ty from './interface';

function Home_Local(){
  const [isLogin, setIsLogin] = useState(true);
  const [username, setUsername] = useState('');
  const [password, setPassword] = useState('');
  const [confirmPassword, setConfirmPassword] = useState('');
  const [session, setSession] = useState<Auth.Session | null>(null);
  const [pageError, setPageError] = useState<string | null>(null);
  const [isErrorInstant, setIsErrorInstant] = useState(false);
  const [pending, setPending] = useState(false);
  const router = useRouter();

  const initSession = async () => {
    //url : /api/auth/startSession
    setSession(null);
    setPageError(null);
    setIsErrorInstant(false);
    try{
      const res = await fetch(`${Ty.urlPrefix}/api/auth/startSession`,  {
        method: 'GET',
      });
      if (!res.ok) {
        throw new Error('无法获取会话信息，错误码：' + res.status);
      }

      const data: Ty.BackendResponse<Auth.Session> = await res.json();
      if (data.code !== 0) {
        throw new Error('会话获取错误，错误信息：' + data.message);
      }

      setSession(data.data);
    } catch (error) {
      console.error(error);
      setSession(null);
      setPageError('无法初始化会话，请重试');
      setIsErrorInstant(true);
    }
  }
  useEffect(() => {
    initSession();
  }, [isLogin]);

  const updatePassword = (value: string, confirm: string | null) => {
    setPassword(value);

    if (value.length < Auth.minPasswordLength) {
      setPageError(`密码长度不能少于 ${Auth.minPasswordLength} 个字符`);
      setIsErrorInstant(true);
    } else if (value.length > Auth.maxPasswordLength) {
      setPageError(`密码长度不能超过 ${Auth.maxPasswordLength} 个字符`);
      setIsErrorInstant(true);
    } else {
      setPageError(null);
      setIsErrorInstant(false);
    }

    if (confirm) {
      if (value !== confirm) {
        setPageError('两次输入的密码不一致');
        setIsErrorInstant(true);
      } else {
        setPageError(null);
        setIsErrorInstant(false);
      }
    }
  }

  const updateConfirmPassword = (password: string, value: string) => {
    setConfirmPassword(value);
    if (password !== value) {
      setPageError('两次输入的密码不一致');
      setIsErrorInstant(true);
    } else {
      setPageError(null);
      setIsErrorInstant(false);
    }
  };

  const handleSubmit = async (asLogin : boolean) => {
    setPending(true);
    setPageError(null);
    setIsErrorInstant(false);
    const Args : Auth.AuthArgs = {
      sessionId: session?.sessionId || '',
      username,
      password,
      userAgent: navigator.userAgent,
      userIP: null // 服务端填充
    };

    try{
      if(asLogin)
      {
        //URL : /api/auth/login
        const res = await fetch(`${Ty.urlPrefix}/api/auth/login`,  {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify(Args),
        });

        if (!res.ok) {
          throw new Error('错误码：' + res.status);
        }

        const data: Ty.BackendResponse<Auth.LoginResult> = await res.json();
        if (data.code !== 0) {
          throw new Error('错误信息：' + data.message);
        }

        setPageError('登录成功，正在跳转……')
        router.push(`/chat?username=${encodeURIComponent(username)}`);
      } else {
        //URL : /api/auth/register
        const res = await fetch(`${Ty.urlPrefix}/api/auth/register`,  {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify(Args),
        });

        if (!res.ok) {
          throw new Error('错误码：' + res.status);
        }

        const data: Ty.BackendResponse<Auth.RegisterResult> = await res.json();
        if (data.code !== 0) {
          throw new Error('错误信息：' + data.message);
        }

        setPageError('注册成功，正在刷新页面……')
        router.refresh();
        window.location.reload();
      }
    } catch (error) {
      console.error(error);
      setPageError((asLogin ? '登录错误！' : '注册错误！') + error.message);
      setIsErrorInstant(false);
    } finally {
      setPending(false);
    }
  };

  return (
    <div className="bg-gray-100 h-screen flex flex-col items-center justify-center relative">
      <h1 className="text-2xl mb-4">请输入用户名</h1>
      <input
        className="border border-gray-300 rounded px-3 py-2 mb-4"
        type="text"
        placeholder="用户名"
        value={username}
        onChange={(e) => setUsername(e.target.value.trim())}
        required
      />
      <input
        className="border border-gray-300 rounded px-3 py-2 mb-4"
        type="password"
        value={password}
        placeholder="密码"
        onChange={(e) => updatePassword(e.target.value.trim(), isLogin ? null : confirmPassword)}
        required
      />
      {!isLogin &&
      <input
        className="border border-gray-300 rounded px-3 py-2 mb-4"
        type="password"
        value={confirmPassword}
        placeholder="确认密码"
        onChange={(e) => updateConfirmPassword(password, e.target.value.trim())}
        required
      />
      }
      <p className="text-red-500 mb-4">{pageError !== null && pageError}</p>
      <div className="flex flex-row">
        <button
          className=
              {username.length && password.length && session && !pending && !(pageError && isErrorInstant) ? 
                "bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600 active:bg-blue-700" :
                "bg-gray-300 text-gray-500 px-4 py-2 rounded cursor-not-allowed"
              }
          onClick={() => handleSubmit(isLogin)}
        >
          {isLogin ? '登录' : '注册'}
        </button>
        <button
          className="
                    bg-blue-500 text-white px-4 py-2 rounded ml-4
                    hover:bg-blue-600 active:bg-blue-700"
          onClick={() => setIsLogin(v => !v)}
        >
          {isLogin ? '切换到注册' : '切换到登录'}
        </button>
      </div>
    </div>
  );

}


function Home_External() {
  const [username, setUsername] = useState('');
  const router = useRouter();

  const handleSubmit = () => {
    if (!username.trim()) return; 
    router.push(`/chat?username=${encodeURIComponent(username)}`);
  };

  return (
    <div className="bg-gray-100 h-screen flex flex-col items-center justify-center">
      <h1 className="text-2xl mb-4">请输入用户名</h1>
      <input
        className="border border-gray-300 rounded px-3 py-2 mb-4"
        placeholder="用户名"
        value={username}
        onChange={(e) => setUsername(e.target.value)}
        
        required
      />
      <button
        className=
            {username.length ? 
                "bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600 active:bg-blue-700" :
                "bg-gray-300 text-gray-500 px-4 py-2 rounded cursor-not-allowed"
            }
        onClick={handleSubmit}
      >
        进入聊天室
      </button>
    </div>
  );
}

const HomeFn = useLocalBackend ? Home_Local : Home_External;
export default function Home() {
  return HomeFn();
}