'use client'
import { useState, useEffect } from 'react'
import * as Ty from '../interface'
import { useRoom } from './room-context'

export default function RoomListProvider() {

    const [roomsLoading, setRoomsLoading] = useState<boolean>(false)
    const [roomsError, setRoomsError] = useState<string>('')
    const [rooms, setRooms] = useState<Ty.RoomPreviewInfo[]>([])
    const { 
        username, 
        currentRoomInfo, 
        setCurrentRoomInfo, 
        msgVersion, 
        triggerRefresh, 
        setShowCreateModal, 
        triggerErrorModal,
        createRoomVersion, 
        newRoomName, 
        setNewRoomName 
    } = useRoom()

    /*
    - url: /api/room/add
    - method: POST
    - argument:
    interface RoomAddArgs {
        user: string;
        roomName: string;
    }
    - response:
    interface RoomAddResult {
        roomId: number;
    }
    */
    const createRoom = async (roomName : string) : Promise<number>  => {

        const args: Ty.RoomAddArgs = {
            user: username,
            roomName
        };
        setRoomsError('');

        try{
            const res = await fetch(Ty.urlPrefix + '/api/room/add', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(args)
            });

            if (!res.ok) {
                throw new Error('创建房间失败，错误码：' + res.status);
            }

            const data: Ty.BackendResponse<Ty.RoomAddResult> = await res.json();

            if (data.code !== 0) {
                throw new Error('创建房间失败，错误信息：' + data.message);
            }

            return data.data.roomId;
        } catch (error) {
            console.error('创建房间失败:', error);
            setRoomsError(error.message);
        }  

        return -1;
    }

    const handleCreate = async () => {
        if (!newRoomName.trim()) return;
        const roomId = await createRoom(newRoomName.trim());
        if (roomId === -1) return;
        setNewRoomName('');
        setShowCreateModal(false);
        triggerRefresh();
        setCurrentRoomInfo({roomId, roomName: newRoomName.trim(), lastMessage: null});
    };

    useEffect(() => {
        handleCreate();
    }, [createRoomVersion]);

    /*
    - url: /api/room/delete
    - method: POST
    - argument:
    interface RoomDeleteArgs {
        user: string;
        roomId: number;
    }
    - response: null （只要code为0即为成功）
    */
   const deleteRoom = async (roomId: number) => {
       const args: Ty.RoomDeleteArgs = {
           user: username,
           roomId
       };
       setRoomsError('');

       try {
           const res = await fetch(Ty.urlPrefix + '/api/room/delete', {
               method: 'POST',
               headers: {
                   'Content-Type': 'application/json'
               },
               body: JSON.stringify(args)
           });

           if (!res.ok) {
               throw new Error('删除房间失败，错误码：' + res.status);
           }

           const data: Ty.BackendResponse<null> = await res.json();

           /*
           Error : You are not the creator. deletion failed.
           {
                "code": 2,
                "message": "auth error: invalid identity",
                "data": null
            }
           */

           if (data.code === 2) {
               triggerErrorModal('你不是房间创建者，无法删除房间。');
               return false;
           }
           else if (data.code !== 0) {
               throw new Error('删除房间失败，错误信息：' + data.message);
           }

           return true;
       } catch (error) {
           console.error('删除房间失败:', error);
           setRoomsError(error.message);
       }

       return false;
   }

   const handleDeleteRoom = async (roomId: number) => {
       const success = await deleteRoom(roomId);
       if (success) {
           if (currentRoomInfo?.roomId === roomId) {
               setCurrentRoomInfo(null);
           }
           triggerRefresh();
       }
   }

    /*
    - url: /api/room/list
    - method: GET
    - argument: null
    - response:
    interface RoomListRes {
        rooms: RoomPreviewInfo[];
    }
    */
    const getRooms = async () => {
        setRoomsLoading(true)
        setRoomsError('')
        try {
        const res = await fetch(Ty.urlPrefix + '/api/room/list', {
            method: 'GET',
        });
        if (!res.ok) {
            throw new Error('无法获取房间列表，错误码：' + res.status);
        }

        const data: Ty.BackendResponse<Ty.RoomListRes> = await res.json();

        if (data.code !== 0) {
            throw new Error('无法获取房间列表，错误信息：' + data.message);
        }

        setRooms(data.data.rooms);

        } catch (error) {
        console.error('房间列表获取失败:', error);
        setRoomsError(error.message);
        } finally {
        setRoomsLoading(false);
        }
    }

    const handleRoomClick = (room: Ty.RoomPreviewInfo) => {
        setCurrentRoomInfo(room);
    }

    useEffect(() => {
        getRooms();

        const interval = setInterval(() => {
        getRooms();
        }, 2000); // 2秒

        // 组件卸载时清除定时器
        return () => {
        clearInterval(interval)
        }
    }, [msgVersion]);

    useEffect(() => {
        if (roomsError) {
            triggerErrorModal(roomsError);
            setRoomsError('');
        }
    }, [roomsError]);

    return (
      <div className="room-container">
        {roomsLoading && !rooms.length && <div className="text-gray-500">加载中...</div>}
        <ul className="mb-2">
          {rooms && rooms.map((room) => (
                <li key={room.roomId} className="relative">
                    <div className="bg-gray-200 p-4 mb-2 border-b border-gray-300 rounded w-full" onClick={() => handleRoomClick(room)}>  
                        <button
                            className="
                            absolute top-1 right-1
                            w-6 h-6
                            bg-red-500 hover:bg-red-600 active:bg-red-700
                            text-white text-sm font-bold
                            rounded
                            flex items-center justify-center
                            z-10
                            "
                            onClick={(e) => {
                            e.stopPropagation(); // 阻止冒泡，避免触发整行点击
                            handleDeleteRoom(room.roomId);
                            }}
                        >
                            —
                        </button>
                        <h3>{room.roomName}</h3>
                        {room.lastMessage ? (
                        <p>{room.lastMessage.sender}: {room.lastMessage.content}</p>
                        ) : (
                        <p className="text-gray-500">暂无消息</p>
                        )}
                    </div>
                </li>
            ))
          }
          <li>
            <button
            className="w-full bg-blue-500 text-white rounded p-4 text-2xl hover:bg-blue-600 active:bg-blue-700"
            onClick={() => setShowCreateModal(true)}
            >
            +
            </button>
          </li>
        </ul>
        
      </div>
  );
}