'use client'
import { useState, useEffect, useRef, use } from 'react'
import * as Ty from './interface'
import { useRoom } from './room-context'

export default function RoomMessageProvider() {

    const { currentRoomInfo, msgVersion } = useRoom()

    const [messagesLoading, setMessagesLoading] = useState<boolean>(false)
    const [messagesError, setMessagesError] = useState<string>('')
    const [messages, setMessages] = useState<Ty.Message[]>([])

    const messagesRef = useRef(messages);
    useEffect(() => { messagesRef.current = messages; }, [messages]);

    /*
    - url: /api/room/message/list
    - method: GET
    - argument:
    interface RoomMessageListArgs {
        roomId: number;
    }
    - response:
    interface RoomMessageListRes {
        messages: Message[];
    }
    */
    const getMessages = async (currentRoomInfo : Ty.RoomPreviewInfo) => {
        setMessagesLoading(true)
        setMessagesError('')

        const Args : Ty.RoomMessageListArgs = {
            roomId: currentRoomInfo.roomId
        }

        const params = new URLSearchParams();
        Object.entries(Args).forEach(([k, v]) => params.set(k, String(v)));

        try {
            const res = await fetch(`${Ty.urlPrefix}/api/room/message/list?${params}`,  {
                method: 'GET',
            });

            if (!res.ok) {
                throw new Error('无法获取房间列表，错误码：' + res.status);
            }

            const data: Ty.BackendResponse<Ty.RoomMessageListRes> = await res.json();

            if (data.code !== 0) {
                throw new Error('无法获取房间列表，错误信息：' + data.message);
            }

            setMessages(data.data.messages);

        } catch (error) {
            console.error('房间列表获取失败:', error);
            setMessagesError(error.message);
        } finally {
            setMessagesLoading(false);
        }
    }

    /*
    - url: /api/room/message/getUpdate
    - method: GET
    - argument:
    interface RoomMessageGetUpdateArgs {
        roomId: number;
        sinceMessageId: number;
    }
    - response:
    interface RoomMessageGetUpdateRes {
        messages: Message[];
    }
    */
    const updateMessages = async (currentRoomInfo : Ty.RoomPreviewInfo, sinceMessageId: number) => {

        setMessagesLoading(true)
        setMessagesError('')

        const Args : Ty.RoomMessageGetUpdateArgs = {
            roomId: currentRoomInfo.roomId,
            sinceMessageId
        }

        const params = new URLSearchParams();
        Object.entries(Args).forEach(([k, v]) => params.set(k, String(v)));

        try{
            const res = await fetch(`${Ty.urlPrefix}/api/room/message/getUpdate?${params}`, {
                method: 'GET',
            });

            if (!res.ok) {
                throw new Error('无法获取房间列表，错误码：' + res.status);
            }

            const data: Ty.BackendResponse<Ty.RoomMessageGetUpdateRes> = await res.json();

            if (data.code !== 0) {
                throw new Error('无法获取房间列表，错误信息：' + data.message);
            }

            setMessages((prevMessages) => [...prevMessages, ...data.data.messages]);

        } catch (error) {
            console.error('房间列表获取失败:', error);
            setMessagesError(error.message);
        } finally {
            setMessagesLoading(false);
        }
    }

    const scrollRef = useRef<HTMLDivElement>(null);

    useEffect(() => {
        setMessagesLoading(false)
        setMessagesError('')
        setMessages([]);

        if(currentRoomInfo)
        {
            getMessages(currentRoomInfo);

            const interval = setInterval(() => {
                const msg = messagesRef.current
                const lastId = msg.length ? msg[msg.length - 1].messageId : 0;
                updateMessages(currentRoomInfo, lastId);
            }, 2000); // 2秒

            return () => {
                clearInterval(interval)
            }
        }
    }, [currentRoomInfo, msgVersion]);

    useEffect(() => {
        if (scrollRef.current) {
            scrollRef.current.scrollTop = scrollRef.current.scrollHeight;
        }
    }, [messages]);

    if(currentRoomInfo)
    {
        return (
            <div
                ref={scrollRef}         
                className="bg-white p-4 overflow-y-auto h-3/5"
            >
                <div className="bg-gray-100 min-h-full">
                    {messagesError && <div className="text-red-500">{messagesError}</div>}
                    <ul className="mb-2">
                        {messages?.map((msg) => (
                        <li key={msg.messageId} className="mb-3">
                            <div className="text-xs text-gray-500 mb-1">{msg.sender}</div>
                            <div className="inline-block max-w-xs lg:max-w-md px-3 py-2 rounded-xl bg-blue-500 text-white">
                            {msg.content}
                            </div>
                        </li>
                        ))}
                    </ul>
                </div>
            </div>
        );
    }
    else
    {
        return (
            <div
                ref={scrollRef}         
                className="bg-white p-4 overflow-y-auto h-3/5"
            >
                <div className="bg-gray-100 min-h-full">
                    <h2 className='text-lg font-bold text-gray-500'>
                        查看消息
                    </h2>
                </div>
            </div>
        );
    }
};