'use client'
import { useState, useEffect } from 'react'
import * as Ty from './interface'
import { useRoom } from './room-context'

export default function RoomListProvider() {

    const [roomsLoading, setRoomsLoading] = useState<boolean>(false)
    const [roomsError, setRoomsError] = useState<string>('')
    const [rooms, setRooms] = useState<Ty.RoomPreviewInfo[]>([])
    const { setCurrentRoomInfo, msgVersion } = useRoom()
    

    /*
    - url: /api/room/list
    - method: GET
    - argument: null
    - response:
    interface RoomListRes {
        rooms: RoomPreviewInfo[];
    }
    */
    const getRooms = async () => {
        setRoomsLoading(true)
        setRoomsError('')
        try {
        const res = await fetch(Ty.urlPrefix + '/api/room/list', {
            method: 'GET',
        });
        if (!res.ok) {
            throw new Error('无法获取房间列表，错误码：' + res.status);
        }

        const data: Ty.BackendResponse<Ty.RoomListRes> = await res.json();

        if (data.code !== 0) {
            throw new Error('无法获取房间列表，错误信息：' + data.message);
        }

        setRooms(data.data.rooms);

        } catch (error) {
        console.error('房间列表获取失败:', error);
        setRoomsError(error.message);
        } finally {
        setRoomsLoading(false);
        }
    }

    const handleRoomClick = (room: Ty.RoomPreviewInfo) => {
        setCurrentRoomInfo(room);
    }

    useEffect(() => {
        getRooms();

        const interval = setInterval(() => {
        getRooms();
        }, 2000); // 2秒

        // 组件卸载时清除定时器
        return () => {
        clearInterval(interval)
        }
    }, [msgVersion]);

    return (
      <div className="room-container">
        {roomsError && <div className="text-red-500">{roomsError}</div>}
        {roomsLoading && !rooms.length && <div className="text-gray-500">加载中...</div>}
        <ul className="mb-2">
          {rooms && rooms.map((room) => (
                <li key={room.roomId} >
                    <div className="bg-gray-200 p-4 mb-2 border-b border-gray-300 rounded w-full" onClick={() => handleRoomClick(room)}>  
                        <h3>{room.roomName} (ID: {room.roomId})</h3>
                        {room.lastMessage ? (
                        <p>{room.lastMessage.sender}: {room.lastMessage.content}</p>
                        ) : (
                        <p>暂无消息。</p>
                        )}
                    </div>
                </li>
            ))
          }
        </ul>
      </div>
  );
}