import { NextResponse } from 'next/server';
import { RequestCookies } from 'next/dist/compiled/@edge-runtime/cookies';
import { createHash, randomBytes } from 'crypto';
import * as Ty from '../interface'
import {prisma, redis} from "@/lib/db";
import * as Auth from './login-interface';
import {z} from "zod";

export async function MakeError(message: string, code: number = 1, status: number = 500): Promise<NextResponse<Ty.BackendResponse<null>>> {
    return NextResponse.json({
        code,
        message,
        data: null
    }, { status });
}

const UserExists = async (name: string): Promise<boolean> => {
    const user = await prisma.user.findUnique({
        where: { name },
        select: { name: true }
    });
    return user !== null;
};

const toFrontMsg = async (m: {
    createdAt: Date;
    id: number;
    content: string;
    roomId: number;
    senderName: string;
}) : Promise<Ty.Message | null> => {
    if(!m.senderName.length) return null;
    if(!await UserExists(m.senderName)) return null;
    return {
       messageId: m.id,
       roomId:    m.roomId,
       sender:    m.senderName,
       content:   m.content,
       time:      m.createdAt.getTime(),
   };
};

const isRoomSelfCreated = async (roomId: number, userToken: string): Promise<boolean> => {
    const room = await prisma.room.findUnique({
        where: { id: roomId },
        select: { creator: { select: { userToken: true } } }
    });
    return room ? room.creator.userToken === userToken : false;
};

const getUserToken = async (name: string) => {
    const user = await prisma.user.findUnique({
        where: { name },
        select: { userToken: true }
    });
    return user ? user.userToken : null;
};

const getUserNameByToken = async (token: string) => {
    const user = await prisma.user.findUnique({
        where: { userToken: token },
        select: { name: true }
    });
    return user ? user.name : null;
};

const getRandomToken = (): string => {
    return [...crypto.getRandomValues(new Uint8Array(16))]
        .map(b => b.toString(36))
        .join('');
};

const getRandomStr = () => (randomBytes(32).toString('hex'));

export async function GetRoomList(userToken : string): Promise<Ty.BackendResponse<Ty.RoomListResEx | null>> {
   const rooms = await prisma.room.findMany({
    select: {
      id: true,
      roomName: true,
      messages: {
        orderBy: { createdAt: 'desc' },
        take: 1
      },
    },
  });
  

  const preview: Ty.RoomPreviewInfoEx[] = await Promise.all(rooms.map(async (r) => {
       const lastMessage = r.messages[0] ? await toFrontMsg(r.messages[0]) : null;
       return {
           roomId: r.id,
           roomName: r.roomName,
           lastMessage,
           selfCreated: await isRoomSelfCreated(r.id, userToken)
       };
   }));

   return {
       code: 0,
       message: '房间获取成功',
       data: {
            rooms: preview
       }
   };
}

export async function AddRoom(data: Ty.RoomAddArgs): Promise<Ty.BackendResponse<Ty.RoomAddResult>> {

    const userName = data.user;
    const creatorToken = await getUserToken(userName);

    if(!creatorToken)
    {
        return {
            code: 1,
            message: '用户不存在',
            data: {
                roomId: -1
            }
        };
    }

    const newRoom = await prisma.room.create({
        data: {
            roomName: data.roomName,
            creatorToken
        }
    });

    return {
        code: 0,
        message: '房间创建成功',
        data: {
            roomId: newRoom.id
        }
    };
}

export async function GetMessageList(data : Ty.RoomMessageListArgs): Promise<Ty.BackendResponse<Ty.RoomMessageListRes>> {
    const { roomId } = data;
    const messages = await prisma.message.findMany({
        where: { roomId },
        orderBy: { createdAt: 'asc' }
    });

    const messageList: Ty.Message[] = 
        await Promise.all(messages.map(toFrontMsg)) 
        .then(msgs => msgs.filter((m): m is Ty.Message => m !== null));

    return {
        code: 0,
        message: '消息获取成功',
        data: {
            messages: messageList
        }
    };
}

export async function UpdateMessageList(data: Ty.RoomMessageGetUpdateArgs): Promise<Ty.BackendResponse<Ty.RoomMessageGetUpdateRes>> {
    const { roomId, sinceMessageId } = data;
    const messages = await prisma.message.findMany({
        where: {
            roomId,
            id: { gt: sinceMessageId }
        },
        orderBy: { createdAt: 'asc' }
    });

    const messageList: Ty.Message[] = 
        await Promise.all(messages.map(toFrontMsg)) 
        .then(msgs => msgs.filter((m): m is Ty.Message => m !== null));

    return {
        code: 0,
        message: '消息更新成功',
        data: {
            messages: messageList
        }
    };
}


export async function DeleteRoom(data: Ty.RoomDeleteArgs, operatorToken: string ): Promise<Ty.BackendResponse<null>> {
    const { user, roomId } = data;

    const room = await prisma.room.findUnique({
        where: { id: roomId },
        select: { creator: { select: { name: true, userToken: true } } }
    });

    if (!room || room.creator.name !== user || room.creator.userToken !== operatorToken) {
        return {
            code: Auth.notAuthorizedToDeleteRoomErrorCode,
            message: '无权限',
            data: null
        };
    }

    await prisma.room.delete({
        where: { id: roomId }
    });

    return {
        code: 0,
        message: '房间删除成功',
        data: null
    };
}

export async function AddMessage(data: Ty.MessageAddArgs): Promise<Ty.BackendResponse<null>> {
    const { roomId, sender, content } = data;

    console.log(`Adding message to room ${roomId} from user ${sender}: ${content}`);

    if (!await UserExists(sender)) {
        return {
            code: 1,
            message: '用户不存在',
            data: null
        };
    }

    await prisma.message.create({
        data: {
            roomId,
            senderName: sender,
            content
        }
    });

    return {
        code: 0,
        message: '消息发送成功',
        data: null
    };
}

export async function StartAuthSession() : Promise<Ty.BackendResponse<Auth.Session>> {
    const sessionId = getRandomToken();

    redis?.set(`Session:${sessionId}`, "{}");
    redis?.expire(`Session:${sessionId}`, 60 * 5); // 5分钟内登录/注册有效

    return {
        code: 0,
        message: '获取成功',
        data: {
            sessionId,
        }
    };
}

export async function Login(args: Auth.AuthArgs) : Promise<Ty.BackendResponse<Auth.LoginResult | null>> {
    const { sessionId, username, password, userAgent, userIP } = args;
    const sessionStr = await redis?.get(`Session:${sessionId}`);
    if(!sessionStr)
    {
        return {
            code: 1,
            message: '会话过期，请刷新页面后重试',
            data: null
        };
    }

    const user = await prisma.user.findUnique({
        where: { name: username },
        select: { password: true, salt: true, userToken: true }
    });

    if(!user || user.password !== createHash('sha256').update(password + user.salt).digest('hex'))
    {
        return {
            code: 3,
            message: '用户名或密码错误',
            data: null
        };
    }

    const authToken = getRandomToken();
    await prisma.authToken.create({
        data: {
            token: authToken,
            userName: username,
            /* 登录过期模拟 ： 改成 + 60000 即可模拟1分钟过期 */
            //expiresAt: new Date(Date.now() + 60000), // 1 minute
            expiresAt: new Date(Date.now() + 60 * 60 * 1000 * 24 * 7), // 7 days
            ua: userAgent,
            ip: userIP
        }
    });

    await redis?.del(`Session:${sessionId}`);

    return {
        code: 0,
        message: '登录成功',
        data: {
            user: username,
            cookie: {
                userToken: user.userToken,
                authToken
            }
        }
    };
}

export async function AutoLogin(args: Auth.AutoLoginArgs) : Promise<Ty.BackendResponse<Auth.LoginResult | null>> {
    const { cookie } = args;
    const { authToken } = cookie;

    const user = await prisma.authToken.findUnique({
        where: { token: authToken },
        select: { user: true, expiresAt: true }
    });

    if (!user) {
        return {
            code: 101,
            message: '无效的用户信息',
            data: null
        };
    }

    return {
        code: 0,
        message: '自动登录成功',
        data: {
            user: user.user.name,
            cookie: {
                userToken: cookie.userToken,
                authToken: cookie.authToken
            }
        }
    };
}

export async function Register(args: Auth.AuthArgs) : Promise<Ty.BackendResponse<Auth.RegisterResult | null>> {

    const { sessionId, username, password } = args;
    const sessionStr = await redis?.get(`Session:${sessionId}`);
    if(!sessionStr)
    {
        return {
            code: 1,
            message: '会话过期，请刷新页面后重试',
            data: null
        };
    }

    const user = await prisma.user.findUnique({
        where: { name: username },
        select: { name: true }
    });
    if(user)
    {
        return {
            code: 3,
            message: '用户名已存在',
            data: null
        };
    }

   const salt = getRandomStr();
   const hashedPassword = createHash('sha256').update(password + salt).digest('hex');

   await prisma.user.create({
       data: {
           name: username,
           password: hashedPassword,
           salt,
           userToken: getRandomToken()
       }
   });

   await redis?.del(`Session:${sessionId}`);

   return {
       code: 0,
       message: '注册成功',
       data: {
           user: username,
       }
   };
}

export async function Logout(args: Auth.LoginCookieData) : Promise<Ty.BackendResponse<null>> {
    const { authToken } = args;

    await prisma.authToken.delete({
        where: { token: authToken }
    });

    return {
        code: 0,
        message: '登出成功',
        data: null
    };
}


export async function VerifyCookie(cookie : RequestCookies) : Promise<Ty.BackendResponse<null> | Auth.LoginCookieData> {
    

    const result = Auth.cookieSchema.safeParse({
            userToken: cookie.get('userToken')?.value || null,
            authToken: cookie.get('authToken')?.value || null
    });
    if(!result.success || !result.data.authToken || !result.data.userToken) {
        return {
            code: Auth.abnormalLoginState,
            message: '登录异常，请重新登录。',
            data: null
        };
    }

    const user = await prisma.authToken.findUnique({
        where: { token: result.data.authToken },
        select: { user: true, expiresAt: true }
    });

    if (!user || user.user.userToken !== result.data.userToken) {
        return {
            code: Auth.invalidUserErrorCode,
            message: '无效的用户信息',
            data: null
        };
    }

    if (user.expiresAt < new Date()) {
        await prisma.authToken.delete({
            where: { token: result.data.authToken }
        });
        return {
            code: Auth.reloginErrorCode,
            message: '登录过期，请重新登录。',
            data: null
        };
    }

    const cookieData: Auth.LoginCookieData = {
            userToken: result.data.userToken,
            authToken: result.data.authToken
    };
    return cookieData;
}

export async function CheckUser(name: string, token: string) : Promise<boolean> {
    const user = await prisma.user.findUnique({
        where: { name },
        select: { userToken: true }
    });
    if (!user) return false;
    return user.userToken === token;
}
export const IncorrectUserResponse = async (token: string) : Promise<NextResponse<Ty.BackendResponse<Ty.UserNameRedirectArgs>>> => {
    return NextResponse.json({
        code: Auth.incorrectUserName,
        message: '用户名与登录令牌不匹配，将重定向到正确用户名。',
        data: { user: await getUserNameByToken(token) }
    });
};

export const IncorrectArgumentResponse  = async (zerr: z.ZodError) : Promise<NextResponse<Ty.BackendResponse<null>>> => {
    console.error(`ZodError: ${zerr}`);
    return NextResponse.json({
        code: Auth.incorrectUserName,
        message: '参数格式错误。: ' + zerr.message,
        data: null
    }, {status : 400});
};

export const CheckLoggedIn = async (token: string, userName: string) : Promise<Ty.BackendResponse<Auth.LoggedInResult | null>> => {
    const authToken = await prisma.authToken.findUnique({
        where: { token },
        select: { user: true, expiresAt: true }
    });

    if (!authToken) return {
        code: Auth.invalidUserErrorCode,
        message: '无效的用户信息，请重新登录。',
        data: null
    }
    if (authToken.expiresAt < new Date()) {
        await prisma.authToken.delete({
            where: { token }
        });
        return {
            code: Auth.reloginErrorCode,
            message: '登录过期，请重新登录。',
            data: null
        };
    }
    if (authToken.user.name !== userName) {
        return {
            code: Auth.incorrectUserName,
            message: '用户名与登录令牌不匹配，将重定向到正确用户名。',
            data: { user: authToken.user.name }
        };
    }

    return {
        code: 0,
        message: '用户已登录',
        data: { user: authToken.user.name }
    };
}