import React from "react";
import '../style/index.css';

/// <reference path="./shim.d.ts" />
/// <reference path="./date-fns.d.ts" />

import { format } from "https://esm.run/date-fns";
import { differenceInDays } from "https://esm.run/date-fns";

/**
 * @param {Date} from
 * @param {Date} to
 * @return {string}
 */
function getTimeDiffString(from, to) {
    if (from > to) {
        alert('Dealine has passed!');
        throw new Error('from must be less than to');
    }
    return differenceInDays(to, from).toLocaleString();
}

/**
 * @param {number} id
 * @param {string} title
 * @param {string} content
 * @param {Date} deadline
 * @param {number} resolved
 */
function addTodoToDOM(id, title, content, deadline, resolved) {
    // bonus: save all todos into localStorage
    let obj = { "title": title, "content": content, "deadline": deadline, "resolved": resolved }
    localStorage.setItem(id.toLocaleString(), JSON.stringify(obj))

    const newId = "todo-" + id;
    const todoElem = document.createElement("div");
    const todoBr = document.createElement("br");
    const todoTitleElem = document.createElement("strong");
    const todoIdElem = document.createElement("span");
    const todoLine = document.createElement("div");
    const todoContentElem = document.createElement("p");
    const todoDeadlineTimeElem = document.createElement("p");
    const todoDeadlineElem = document.createElement("i");
    const todoNextElem = document.createElement("p");

    todoElem.id = newId;
    todoElem.className = "todo-record";
    todoLine.className = "todo-line";
    todoElem.setAttribute("x-resolved", resolved.toLocaleString());
    if (resolved) {
        todoElem.style.color = "gray";
    }
    todoTitleElem.id = "title" + id;
    todoTitleElem.className = "todo-record-title";
    todoContentElem.id = "content" + id;
    todoContentElem.className = "todo-record-content";
    todoDeadlineTimeElem.id = "deadline" + id;
    todoDeadlineElem.id = "time" + id;

    todoIdElem.textContent = " (id=" + id + ") ";
    todoTitleElem.textContent = title;
    todoContentElem.textContent = content;
    todoDeadlineTimeElem.textContent = "Deadline: " + format(deadline, "yyyy-MM-dd");
    todoDeadlineElem.textContent = "Remaining time: " + getTimeDiffString(new Date, deadline) + " days";
    // todoElem.appendChild(todoBr);
    todoElem.appendChild(todoNextElem);
    todoElem.appendChild(todoTitleElem);
    todoElem.appendChild(todoIdElem);
    todoElem.appendChild(todoLine);
    todoElem.appendChild(todoContentElem);
    todoElem.appendChild(todoDeadlineTimeElem);
    todoElem.appendChild(todoDeadlineElem);
    todoElem.appendChild(todoNextElem);

    const parentElem = document.getElementById("list-todo");
    if (!parentElem) {
        alert("Failed to get input element");
        return;
    }
    parentElem.appendChild(todoElem);
    parentElem.appendChild(todoBr);
}

/**
 * @param {string} id
 */
function updateOverdueStatus(id) {
    const todo = document.getElementById("todo-" + id);
    if (!todo) {
        return;
    }
    if (todo.getAttribute("resolved") === "1") {
        return;
    }

    const todoDeadlineElemSet = todo.getElementsByTagName("p");  // ref: index.html example todo div
    /*  if (todoDeadlineElemSet.length !== 1) {
        alert("Failed to get deadline element: unexpected number of <???> elements");
      }*/
    const todoDeadline = todoDeadlineElemSet[1].textContent;  // get text content of element
    if (!todoDeadline) {
        return;
    }
    const deadline = getTimeDiffString(new Date, new Date(todoDeadline)); // use date-fns here to parse date string
    if (parseInt(deadline) < 10) {
        const todoTime = todo.getElementsByClassName("");
        todo.style.color = "red";  // set color to red
    }
}

/**
 * @param {string} id 
 * @param {string} part 
 * @param {string} newContent 
 */
function modifyTodoDOM(id, part, newContent) {
    if (localStorage.hasOwnProperty(id)) {
        const todo = document.getElementById(part + id);
        const getVal = JSON.parse(localStorage.getItem(id) || '0');
        if (!todo) {
            alert('Id or element does not exist!');
            return;
        }

        if (part === "title") {
            getVal.title = newContent;
            todo.textContent = newContent;
            // 错误写法：todo?.textContent -> 赋值表达式的左侧不能是可选属性访问
        } else if (part === "content") {
            getVal.content = newContent;
            todo.textContent = newContent;
        } else if (part === "deadline") {
            getVal.deadline = newContent;
            todo.textContent = "Deadline: " + newContent
            const remain = document.getElementById("time" + id);
            if (remain) {
                remain.textContent = "Remaining time: " + getTimeDiffString(new Date, new Date(newContent)) + " days";
            }
        }
        let obj = { "title": getVal.title, "content": getVal.content, "deadline": getVal.deadline, "resolved": getVal.resolved }
        localStorage.setItem(id, JSON.stringify(obj))
        updateOverdueStatus(id);
    } else {
        alert(id);
        alert('Id does not exist!');
        return;
    }

}

/**
 * @param {string} id
 */
function resolveTodo(id) {
    const todo = document.getElementById("todo-" + id);
    if (!todo) {
        return;
    }

    todo.setAttribute("x-resolved", "1");
    todo.style.color = "gray";  // set color to gray

    const getVal = JSON.parse(localStorage.getItem(id) || '0');
    let obj = { "title": getVal.title, "content": getVal.content, "deadline": getVal.deadline, "resolved": 1 }
    localStorage.setItem(id, JSON.stringify(obj))
}

/**
 * @param {string} id 
 */
function deleteTodoDOM(id) {
    const todo = document.getElementById("todo-" + id);
    localStorage.removeItem(id);
    window.location.reload();
}

const addTodoRegex = /^add(\s+)([\w\W]*)(\s+)([\w\W]*)(\s+)(\d{4}[\-\/](\d{2}|\d{1})[\-\/](\d{2}|\d{1}))$/;
const modifyTodoRegex = /^modify(\s+)(\d+)(\s+)(\w+)[(:)(=)(\s+)]([\w\W]*)$/;
const resolveTodoRegex = /^resolve(\s+)(\d+)/;
const deleteTodoRegex = /^delete(\s+)(\d+)/;

class Button extends React.Component {
    constructor() {
        super();
        this.handleClick = this.handleClick.bind(this);
    }

    handleClick() {
        const shellInputElem = document.getElementById('input-command');
        let inputContent = shellInputElem.value;
        inputContent = inputContent.replace(':', ' ');
        inputContent = inputContent.replace('=', ' ');
        if (addTodoRegex.test(inputContent)) {
            const str = inputContent.split(/\s+/);
            addTodoToDOM(++id, str[1], str[2], new Date(str[3]), 0);
        } else if (modifyTodoRegex.test(inputContent)) {
            const str = inputContent.split(/\s+/);
            modifyTodoDOM(str[1], str[2], str[3]);
        } else if (deleteTodoRegex.test(inputContent)) {
            const str = inputContent.split(/\s+/);
            deleteTodoDOM(str[1]);
        } else if (resolveTodoRegex.test(inputContent)) {
            const str = inputContent.split(/\s+/);
            resolveTodo(str[1]);
        } else {
            alert('Invalid Command!');
        }
        shellInputElem.value = '';
    }

    render() {
        return (
            <button id="btn-submit" onClick={this.handleClick}>Submit</button>
        );
    }
}

class Input extends React.Component {
    constructor() {
        super();
        this.handleKeyUp = this.handleKeyUp.bind(this);
    }

    handleKeyUp = e => {
        if (e.keyCode === 13) {
            const shellInputElem = document.getElementById('input-command');
            let inputContent = shellInputElem.value;
            inputContent = inputContent.replace(':', ' ');
            inputContent = inputContent.replace('=', ' ');

            if (addTodoRegex.test(inputContent)) {
                const str = inputContent.split(/\s+/);
                addTodoToDOM(++id, str[1], str[2], new Date(str[3]), 0);
                updateOverdueStatus(id.toString());
            } else if (modifyTodoRegex.test(inputContent)) {
                const str = inputContent.split(/\s+/);
                modifyTodoDOM(str[1], str[2], str[3]);
            } else if (deleteTodoRegex.test(inputContent)) {
                const str = inputContent.split(/\s+/);
                deleteTodoDOM(str[1]);
            } else if (resolveTodoRegex.test(inputContent)) {
                const str = inputContent.split(/\s+/);
                resolveTodo(str[1]);
            } else {
                alert('Invalid Command!');
            }
            shellInputElem.value = '';
        }
    }

    render() {
        return (
            <input className="todo-header-input" type="text" id="input-command" required="" onKeyUp={this.handleKeyUp}></input>
        );        
    }
}

let id = 0;
export default function Wrapper() {
    let res = [];
    for (let i = 0; i < localStorage.length; i++) {
        const getKey = localStorage.key(i);
        if (getKey != null) {
            let getVal = JSON.parse(localStorage.getItem(getKey) || '0');
            // res.push(getVal); // 可能导致 reload后乱序问题
            res[parseInt(getKey) - 1] = getVal
        }
    }
    localStorage.clear();
    for (let i = 0; i < res.length; i++) {
        let obj = res[i];
        if (obj) {
            addTodoToDOM(++id, obj.title, obj.content, new Date(obj.deadline), parseInt(obj.resolved));
            updateOverdueStatus(id.toLocaleString());
        }
    }

    return (
        <div>
            <div className="todo-header">
                <Input />
                <div className="underline"></div>
                <label>Command</label>
            </div>
            <Button />
        </div>
    );
}

