// @ts-check

/// <reference path="./shim.d.ts" />
/// <reference path="./date-fns.d.ts" />

import { format, compareAsc, parse } from 'https://esm.run/date-fns';

/**
 * @param {Date} from
 * @param {Date} to
 * @return {string}
 */
function getTimeDiffString(from, to) {
  if(from > to) {
    throw new Error('from must be less than to');
  }
  // use date-fns here
  return "not implemented";
}

var id = 1;

/**
 * @param {number} id
 * @param {string} title
 * @param {string} content
 * @param {Date} deadline 
 */
function addTodoToDOM(id, title, content, deadline, resolve = 0) {
  // bonus: save all todos into localStorage

  const newId = "todo-" + id;
  const todoElem = document.createElement("div");
  const todoTitleElem = document.createElement("strong");
  const todoIdElem = document.createElement("span");
  const todoContentElem = document.createElement("p");
  const todoDeadlineElem = document.createElement("i");

  todoElem.id = newId;
  todoTitleElem.textContent = title;
  todoIdElem.textContent = "(id=" + id + ")";
  todoContentElem.textContent = content;
	const formatdate = format(deadline, 'yyyy/MM/dd');
	todoDeadlineElem.textContent = formatdate;

  todoElem.appendChild(todoTitleElem);
  todoElem.appendChild(todoIdElem);
	todoElem.appendChild(todoContentElem);
	todoElem.appendChild(todoDeadlineElem);

  const parentElem = document.getElementById("list-todo");
  if (!parentElem) {
    alert("Failed to get input element");
    return;
  }
  parentElem.appendChild(todoElem);

  localStorage.setItem("todo-" + id, JSON.stringify({ "title": title, "content": content, "deadline": formatdate, "resolve": resolve}));
}

/**
 * @param {number} id
 */
function resolveTodo(id) {
  const todo = document.getElementById("todo-" + id);
  if (!todo) {
    return;
  }

  todo.setAttribute("x-resolved", "1");
  todo.style.color = "gray";  // set color to gray

  let item = JSON.parse(localStorage.getItem("todo-" + id) || '0');
  localStorage.setItem("todo-" + id, JSON.stringify({"title": item.title, "content": item.content, "deadline": item.deadline, "resolve": 1}));
}

function deleteTodo(id) {
  const todo = document.getElementById("todo-" + id);
  if(!todo) {
    return;
  }

  if(todo.parentNode)
  {
    todo.parentNode.removeChild(todo);
  }

  localStorage.removeItem("todo-" + id);
}

/**
 * @param {number} id
 */
function updateOverdueStatus(id) {
  const todo = document.getElementById("todo-" + id);
  if (!todo) {
    return;
  }
  if (todo.getAttribute("x-resolved") === "1") {
    return;
  }

  var currentDate = new Date();
  const todoDeadlineElemSet = todo.getElementsByTagName("i");  // ref: index.html example todo div
  if (todoDeadlineElemSet.length !== 1) {
    alert("Failed to get deadline element: unexpected number of <i> elements");
  }
  const todoDeadline = todoDeadlineElemSet[0].textContent;  // get text content of element
  if(!todoDeadline) {
    return;
  }
  const deadline = parse(todoDeadline, 'yyyy/MM/dd', currentDate); // use date-fns here to parse date string
  if (compareAsc(deadline, currentDate) === -1) {
    todo.style.color = "red";  // set color to red
  }
}

function initializeTodoApp() {
  // bonus: load all saved todos from localStorage

  if(localStorage.getItem("todo-0")) { // 取回localstorage中存储的全局变量id
    let item = JSON.parse(localStorage.getItem("todo-0") || '0');
    id = parseInt(item.id);
  }
  for(let i = 1; i < id; i++) {
    let elem = localStorage.getItem("todo-" + i);
    if(elem) {
      let item = JSON.parse(elem);
      addTodoToDOM(i, item.title, item.content, parse(item.deadline, 'yyyy/MM/dd', new Date()), item.resolve);
      if(item.resolve) {
        resolveTodo(i);
      }
      else {
        updateOverdueStatus(i);
      }
    }
  }

  const addTodoRegex = /^add\s+(\w+)\s+(\w+)\s+(\d+)\/(\d+)\/(\d+)$/;
  const modifyTodoRegex = /^modify\s+(\d+)\s+(\w+)\s*=\s*(.+)/;
  const resolveTodoRegex = /^resolve\s+(\d+)\s*/;
  const deleteTodoRegex = /^delete\s+(\d+)\s*/;
  const updateTodoRegex = /^update\s*/;

  const shellInputElem = /** @type {HTMLInputElement | null} */ (document.getElementById("input-todo"));
  if (shellInputElem === null) {
    alert("Failed to get input element");
    return;
  }
  // set event listener
  shellInputElem.addEventListener("keyup", function (ev) {
    if (ev.key === "Enter") {
      const inputContent = shellInputElem.value;
      if (addTodoRegex.test(inputContent)) {
        let match = inputContent.match(addTodoRegex);
        var deadline = new Date();
        if(!match || match.length !== 6) // immediately test the possible NULL variable
        {
          alert("add operation input format is wrong");
          return;
        }
        deadline.setFullYear(parseInt(match[3]), parseInt(match[4]) - 1, parseInt(match[5]));
        addTodoToDOM(id++, match[1], match[2], deadline);

        localStorage.setItem("todo-0", JSON.stringify({"id": id})); // 每次修改id，同时在localstorage对应位置修改id的值
      } 
      else if (modifyTodoRegex.test(inputContent)) {
        let match = inputContent.match(modifyTodoRegex);
        if(!match || match.length !== 4)
        {
          alert("modify operation input format is wrong");
          return;
        }
        match[3] = match[3].trim(); // delete the front and back spaces
        var modifyElem = document.getElementById("todo-" + match[1]);
        if(!modifyElem) {
          alert("Failed to get modify element");
          return; // to prevent future error
        }
        match[2].toLocaleLowerCase(); // upper and lower input texts are both supported
        if(match[2] === "title") {
          const elem = modifyElem.getElementsByTagName("strong"); // getElementByTagName returns an array, so elem should be used with index
          elem[0].textContent = match[3];

          let item = JSON.parse(localStorage.getItem("todo-" + match[1]) || '0');
          localStorage.setItem("todo-" + match[1], JSON.stringify({"title": match[3], "content": item.content, "deadline": item.deadline, "resolve": item.resolve}));
        }
        else if(match[2] === "content") {
          const elem = modifyElem.getElementsByTagName("p");
          elem[0].textContent = match[3];

          let item = JSON.parse(localStorage.getItem("todo-" + match[1]) || '0');
          localStorage.setItem("todo-" + match[1], JSON.stringify({"title": item.title, "content": match[3], "deadline": item.deadline, "resolve": item.resolve}));
        } else if(match[2] === "deadline"){
            const elem = modifyElem.getElementsByTagName("i");
            var date = match[3].split("/");
            if(date.length !== 3)
            {
                alert("duedate input format is wrong");
                return;
            }
            date[0] = date[0].trim(); // spaces are allowed in input date format " yyyy / MM  / dd "
            date[1] = date[1].trim();
            date[2] = date[2].trim();
            elem[0].textContent = date[0] + "/" + date[1] + "/" + date[2];

            let item = JSON.parse(localStorage.getItem("todo-" + match[1]) || '0');
            localStorage.setItem("todo-" + match[1], JSON.stringify({"title": item.title, "content": item.content, "deadline": elem[0].textContent, "resolve": item.resolve}));
        }
      }
      else if(resolveTodoRegex.test(inputContent)) {
        let match = inputContent.match(resolveTodoRegex);
        if(!match || match.length !== 2)
        {
          alert("Failed to get resolve element");
          return;
        }
        resolveTodo(parseInt(match[1]));
      }
      else if(deleteTodoRegex.test(inputContent)) {
        let match = inputContent.match(deleteTodoRegex);
        if(!match || match.length !== 2)
        {
          alert("Failed to get delete element");
          return;
        }
        deleteTodo(parseInt(match[1]));
      }
      else if(updateTodoRegex.test(inputContent)) {
        for(var i = 1; i < id; i++) {
          updateOverdueStatus(i);
        }
      }
      else {
        var s = "operation \"" + inputContent.split(" ")[0] + "\" is not supported";
        alert(s);
      }
    }
  })

  // or: btnElem.addEventListener("click", function () {})
}

window.addEventListener("load", initializeTodoApp);
