// app/login/page.tsx (或者你存放此组件的文件)

'use client';

import React, { useState } from 'react';
import { useRouter } from 'next/navigation'; // 引入 useRouter 用于页面跳转
import './page.css'; // 全局样式文件

export default function LoginPage() {
  const router = useRouter(); // 初始化 router
  const [credentials, setCredentials] = useState({
    username: '',
    password: ''
  });
  
  // 新增状态：用于处理加载和错误信息
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value } = e.target;
    setCredentials(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsLoading(true); // 开始请求，进入加载状态
    setError(null);     // 清除之前的错误信息

    try {
      const response = await fetch('/api/auth/login', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(credentials),
      });

      const data = await response.json();

      if (!response.ok) {
        // 如果服务器返回错误状态码 (e.g., 400, 401, 500)
        // 从返回的json中获取错误消息
        throw new Error(data.message || '登录失败，请稍后重试');
      }

      // 登录成功
      console.log('登录成功:', data);
      // alert('登录成功！即将跳转到主页...'); // 可以给用户一个提示
      
      // 重定向到聊天室列表页或仪表盘
      // 假设你的聊天室主页是 '/'
      router.push(`/chatroom?username=${encodeURIComponent(credentials.username)}`); 

    } catch (err) {
      // 捕获网络错误或上面抛出的错误
      console.error('登录请求失败:', err);
      if (err instanceof Error) {
        setError(err.message);
      } else {
        setError('发生未知错误');
      }
    } finally {
      setIsLoading(false); // 请求结束，无论成功与否，都结束加载状态
    }
  };

  return (
    <div className="login-page">
      {/* 动态背景元素 */}
      <div className="dynamic-bg">
        <div className="gradient-bg"></div>
        <div className="particles">
          {[...Array(15)].map((_, i) => <div key={i} className="particle"></div>)}
        </div>
      </div>
      
      {/* 登录表单 */}
      <div className="login-container">
        <form onSubmit={handleSubmit} className="login-form">
          <h1 className="login-title">用户登录</h1>
          
          <div className="input-group">
            <label htmlFor="username" className="input-label">用户名</label>
            <input
              type="text"
              id="username"
              name="username"
              value={credentials.username}
              onChange={handleChange}
              className="form-input"
              placeholder="请输入用户名"
              required
              disabled={isLoading} // 加载时禁用输入框
            />
          </div>
          
          <div className="input-group">
            <label htmlFor="password" className="input-label">密码</label>
            <input
              type="password"
              id="password"
              name="password"
              value={credentials.password}
              onChange={handleChange}
              className="form-input"
              placeholder="请输入密码"
              required
              disabled={isLoading} // 加载时禁用输入框
            />
          </div>

          {/* 错误信息显示区域 */}
          {error && <p className="error-message">{error}</p>}
          
          <button type="submit" className="submit-button" disabled={isLoading}>
            {isLoading ? '登录中...' : '登录'}
          </button>
          
          <div className="login-footer">
            <a href="#" className="forgot-password">忘记密码?</a>
            <a href="#" className="signup-link">注册账号</a>
          </div>
        </form>
      </div>
    </div>
  );
}