// app/api/auth/login/route.ts

import { NextResponse } from 'next/server';
import { findUserByUsername, createUser } from '@/server/db'; // 确保路径正确
import bcrypt from 'bcryptjs';
import jwt from 'jsonwebtoken';

interface ApiResponse {
  code: number;
  message?: string;
}

export async function POST(request: Request) {
  try {
    const { username, password } = await request.json();

    // 1. 输入验证
    if (!username || !password) {
      return NextResponse.json({ message: '用户名和密码不能为空' }, { status: 400 });
    }

    // 2. 在数据库中查找用户
    let user = await findUserByUsername(username);

    if (user) {
      // 策略2: 账号存在，检验密码
      const isPasswordCorrect = await bcrypt.compare(password, user.passwordHash);
      if (!isPasswordCorrect) {
        // 为了安全，不要明确提示是密码错了还是用户名错了
        return NextResponse.json({ message: '用户名或密码错误' }, { status: 401 });
      }
    } else {
      // 策略1: 账号不存在，自动创建账号
      // a. 哈希密码
      const saltRounds = 10; // 推荐的哈希计算强度
      const passwordHash = await bcrypt.hash(password, saltRounds);
      
      // b. 创建用户
      const newUser = await createUser(username, passwordHash);
      user = { userId: newUser.userId, username: newUser.username, passwordHash: '', createdAt: new Date() }; // 组装一个临时的 user 对象用于生成 token
    }

    // 3. 登录成功，生成 JWT
    const secret = process.env.JWT_SECRET;
    if (!secret) {
        throw new Error('JWT_SECRET is not defined in environment variables');
    }

    const payload = {
      userId: user.userId,
      username: user.username,
    };
    
    // Token 有效期，例如 1 小时
    const token = jwt.sign(payload, secret, { expiresIn: '1h' });

    // 4. 将 Token 放入 httpOnly Cookie 中返回给客户端
    const response = NextResponse.json({ 
        success: true, 
        message: '登录成功',
        user: { userId: user.userId, username: user.username } // 可以选择性返回一些用户信息
    });
    
    response.cookies.set('token', token, {
      httpOnly: true, // 关键！防止客户端JS读取，可防御XSS攻击
      secure: process.env.NODE_ENV === 'production', // 仅在 HTTPS 下发送
      sameSite: 'strict', // 防止CSRF攻击
      maxAge: 60 * 60, // 1小时，与token有效期一致
      path: '/', // Cookie作用于整个网站
    });

    return response;

  } catch (error) {
    console.error('Login API error:', error);
    return NextResponse.json({ message: '服务器内部错误' }, { status: 500 });
  }
}

// 处理其他不允许的方法
export async function GET() {
  return NextResponse.json({
    code: 1,
    message: 'Method not allowed. Use POST instead'
  } satisfies ApiResponse, {
    status: 405
  });
}

export async function PUT() {
  return NextResponse.json({
    code: 1,
    message: 'Method not allowed'
  } satisfies ApiResponse, {
    status: 405
  });
}

export async function DELETE() {
  return NextResponse.json({
    code: 1,
    message: 'Method not allowed'
  } satisfies ApiResponse, {
    status: 405
  });
}