"use client";

import React, { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import { RoomPreviewInfo, Message } from "@/types/chat";
import RoomEntry from "@/components/RoomEntry";
import MessageItem from "@/components/MessageItem";
import { addRoom, getRooms, deleteRoom } from "@/lib/api";

export default function ChatRoomPage() {
  const [rooms, setRooms] = useState<RoomPreviewInfo[]>([]);
  const [currentRoomId, setCurrentRoomId] = useState<number | null>(null);
  const [messages, setMessages] = useState<Message[]>([]);
  const [newMessage, setNewMessage] = useState("");
  const [currentUser, setCurrentUser] = useState<string | null>(null);
  const [mounted, setMounted] = useState(false);

  const router = useRouter();

  const getMessage = async () => {
    const res = await getRooms();
    return res.data.rooms.map((r: any) => ({
            roomId: r.id,
            roomName: r.roomName,
            lastMessage: {
              roomid: r.id,
              messageId: r.lastMessage?.id || null,
              content: r.lastMessage?.content || "",
              sender: r.lastMessage?.sender || "",
              time: r.lastMessage?.time || new Date().toISOString(),
            },
      }))
  };

  // ✅ 挂载时获取用户名 & 房间列表
  useEffect(() => {
    setMounted(true);
    const username = localStorage.getItem("username");
    if (username) {
      setCurrentUser(username);
    }
    async function fetchRooms() {
      const res = await getMessage();
      setRooms(res);
    }
    fetchRooms();
  }, []);

  // ✅ 房间消息更新轮询
  useEffect(() => {
    if (!currentRoomId) return;

    const interval = setInterval(async () => {
      if (messages.length === 0) return;

      const lastId = messages[messages.length - 1].messageId;

      try {
        const res = await fetch(
          `/api/room/message/getUpdate?roomId=${currentRoomId}&sinceMessageId=${lastId}`
        );
        const data = await res.json();
        if (data.code === 0 && data.data.messages.length > 0) {
          setMessages((prev) => [...prev, ...data.data.messages]);
        }
      } catch (err) {
        console.error("更新消息失败:", err);
      }
    }, 3000);

    return () => clearInterval(interval);
  }, [currentRoomId, messages]);

  // ✅ 添加房间
  const handleAddRoom = async () => {
    if (!currentUser) return alert("请先登录！");
    const name = prompt("请输入新房间名称");
    if (!name) return;

    const res = await addRoom(currentUser, name);
    if (res.code === 0) {
      const listRes = await getMessage();
      setRooms(listRes);
    }
  };
  // ✅ 删除房间
  const handleDeleteRoom = async (roomId: number) => {
    if (!currentUser) return alert("请先登录！");
    const confirmDelete = confirm("确定要删除这个房间吗？");
    if (!confirmDelete) return;

    const res = await deleteRoom(currentUser, roomId);
    if (res.code === 0) {
      const listRes = await getMessage();
      setRooms(listRes);
      if (currentRoomId === roomId) {
        setCurrentRoomId(null);
        setMessages([]);
      }
    } else {
      alert("删除失败：" + res.error);
    }
  };

  // ✅ 选择房间
  const handleSelectRoom = async (roomId: number) => {
    setCurrentRoomId(roomId);

    const res = await fetch(`/api/room/message/list?roomId=${roomId}`);
    const data = await res.json();

    if (data.code === 0) {
      setMessages(data.data.messages);
    } else {
      setMessages([]);
    }
  };

  // ✅ 发送消息
  const handleSendMessage = async () => {
    if (!newMessage.trim() || currentRoomId === null || !currentUser) return;

    const now = Date.now();
    const optimisticMsg: Message = {
      messageId: now,
      roomId: currentRoomId,
      sender: currentUser,
      content: newMessage,
      time: now,
    };

    setMessages((prev) => [...prev, optimisticMsg]);
    setRooms((prev) =>
      prev.map((room) =>
        room.roomId === currentRoomId ? { ...room, lastMessage: optimisticMsg } : room
      )
    );
    setNewMessage("");

    try {
      const res = await fetch("/api/message/add", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          roomId: Number(currentRoomId),
          content: newMessage,
          sender: currentUser,
        }),
      });
      const data = await res.json();
      if (data.code !== 0) {
        console.error("发送失败:", data.error);
      }
    } catch (err) {
      console.error("发送消息异常:", err);
    }
  };

  if (!mounted) return <div>加载中...</div>;

return (
    <div className="flex flex-col h-screen bg-gray-50">
      {/* 顶部用户栏 */}
      <div className="flex justify-between items-center px-6 py-3 border-b border-gray-200 bg-white shadow-sm">
        <div className="flex items-center">
          <span className="text-gray-700 font-medium">当前用户：</span>
          <span className="ml-2 text-indigo-600 font-semibold">
            {currentUser ?? <span className="text-red-500">未登录</span>}
          </span>
        </div>
        {currentUser && (
          <button
            onClick={() => {
              localStorage.removeItem("username");
              setCurrentUser(null);
              router.push("/login");
            }}
            className="px-3 py-1.5 bg-red-500 text-white rounded-md hover:bg-red-600 transition duration-200 flex items-center text-sm"
          >
            <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4 mr-1" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1" />
            </svg>
            退出
          </button>
        )}
      </div>

      {/* 主体内容 */}
      <div className="flex flex-1 overflow-hidden">
        {/* 左侧房间列表 */}
        <div className="w-64 border-r border-gray-200 bg-white flex flex-col">
          <div className="p-3 border-b border-gray-200">
            <button 
              onClick={handleAddRoom}
              className="w-full px-3 py-2 bg-indigo-600 text-white rounded-md hover:bg-indigo-700 transition duration-200 flex items-center justify-center text-sm"
            >
              <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4 mr-1" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 6v6m0 0v6m0-6h6m-6 0H6" />
              </svg>
              添加房间
            </button>
          </div>
          <div className="flex-1 overflow-y-auto">
            {rooms.map((room) => (
              <div
                key={room.roomId}
                className={`flex items-center p-2 border-b border-gray-100 hover:bg-gray-50 transition duration-150 ${
                  room.roomId === currentRoomId ? "bg-blue-50 border-l-4 border-l-blue-500" : ""
                }`}
              >
                <div className="flex-1 min-w-0">
                  <RoomEntry
                    room={room}
                    isActive={room.roomId === currentRoomId}
                    onClick={() => handleSelectRoom(room.roomId)}
                    onDelete={() => handleDeleteRoom(room.roomId)}
                  />
                </div>
                <button
                  onClick={() => handleDeleteRoom(room.roomId)}
                  className="ml-1 p-1 text-gray-400 hover:text-red-500 transition duration-150 rounded-full hover:bg-red-50"
                  title="删除房间"
                >
                  <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                  </svg>
                </button>
              </div>
            ))}
          </div>
        </div>

        {/* 右侧聊天窗口 */}
        <div className="flex-1 flex flex-col bg-white">
          <div className="flex-1 p-4 overflow-y-auto bg-gray-50">
            {messages.length === 0 ? (
              <div className="h-full flex items-center justify-center">
                <div className="text-center text-gray-400">
                  <svg xmlns="http://www.w3.org/2000/svg" className="h-12 w-12 mx-auto mb-3" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 10h.01M12 10h.01M16 10h.01M9 16H5a2 2 0 01-2-2V6a2 2 0 012-2h14a2 2 0 012 2v8a2 2 0 01-2 2h-5l-5 5v-5z" />
                  </svg>
                  <p className="text-sm">选择一个房间开始聊天</p>
                </div>
              </div>
            ) : (
              messages.map((msg) => (
                <MessageItem key={msg.time} message={msg} />
              ))
            )}
          </div>
          {currentRoomId && (
            <div className="p-3 border-t border-gray-200 bg-white">
              <div className="flex space-x-2">
                <input
                  value={newMessage}
                  onChange={(e) => setNewMessage(e.target.value)}
                  className="flex-1 px-3 py-2 border border-gray-300 rounded-full focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 text-sm"
                  placeholder="输入消息..."
                  onKeyPress={(e) => {
                    if (e.key === 'Enter') {
                      handleSendMessage();
                    }
                  }}
                />
                <button
                  onClick={handleSendMessage}
                  disabled={!newMessage.trim()}
                  className={`px-3 py-2 rounded-full flex items-center text-sm ${
                    newMessage.trim() 
                      ? "bg-indigo-600 text-white hover:bg-indigo-700" 
                      : "bg-gray-200 text-gray-400 cursor-not-allowed"
                  } transition duration-200`}
                >
                  <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4 mr-1" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 19l9 2-9-18-9 18 9-2zm0 0v-8" />
                  </svg>
                  发送
                </button>
              </div>
            </div>
          )}
        </div>
      </div>
    </div>
  );
}
